package org.mulesoft.apb.project.internal.engine

import amf.aml.client.scala.model.document.{Dialect, DialectInstance}
import amf.core.client.scala.AMFParseResult
import amf.core.client.scala.resource.ResourceLoader
import amf.shapes.client.scala.model.document.JsonSchemaDocument
import org.mulesoft.apb.project.client.scala.dependency._
import org.mulesoft.apb.project.client.scala.descriptor.DescriptorParseResult
import org.mulesoft.apb.project.client.scala.environment.DependencyFetcher
import org.mulesoft.apb.project.client.scala.model.descriptor.{
  DependencyScope,
  ExtensionScope,
  Gav,
  ManagementScope,
  ProjectDependency,
  ProjectDescriptor,
  ValidationScope
}
import org.mulesoft.apb.project.client.scala.model.report.APBResult
import org.mulesoft.apb.project.client.scala.ProjectConfiguration

import scala.concurrent.ExecutionContext.Implicits.global
import scala.concurrent.Future

case class AnypointTreeBuilder(
    dependencyFetcher: DependencyFetcher,
    loaders: List[ResourceLoader],
    cacheBuilder: UnitCacheBuilder
) extends TreeBuilderTemplate(dependencyFetcher, loaders, cacheBuilder)
    with CollisionComputationHelper {

  override def build(project: ProjectDescriptor): Future[ProjectConfiguration] =
    super
      .build(project)
      .flatMap { lookupRisksForRoot(_, loaders) }
      .map { assignMigrationRisks }

  private def processDependency(
      descriptorParse: DescriptorParseResult,
      dependency: ProjectDependency,
      dependenciesInPath: Set[Gav],
      loader: ResourceLoader,
      parentClassifier: Option[String]
  ): Future[ParsedDependency] = {
    val DescriptorParseResult(descriptor, results) = descriptorParse
    for {
      config       <- buildProjectConfig(descriptor, dependenciesInPath + descriptor.gav)
      mainFileInfo <- MainFileInfoBuilder.build(loader, descriptor, config, parentClassifier, cacheBuilder)
      parsedDependency <- {
        val descriptorErrors = setDependencyGavAsLocation(dependency, results.toList)
        val errors           = computeNextErrors(config.results, mainFileInfo.result, descriptorErrors)
        buildParsedDependency(mainFileInfo, errors, dependency.scope, config.migrationRisks)
      }
    } yield parsedDependency
  }

  protected override def parseDependency(
      descriptorParse: DescriptorParseResult,
      dependency: ProjectDependency,
      dependenciesInPath: Set[Gav],
      loader: ResourceLoader,
      parentClassifier: Option[String]
  ): Future[ParsedDependency] = {
    for {
      dependency       <- processDependency(descriptorParse, dependency, dependenciesInPath, loader, parentClassifier)
      mergedDependency <- MigrationRisk.compute(dependency, loader)
    } yield {
      mergedDependency
    }
  }

  private def computeNextErrors(
      results: Seq[APBResult],
      mainFile: AMFParseResult,
      descriptorErrors: List[APBResult]
  ) = {
    results ++ mainFile.results.map(APBResult.forContract) ++ descriptorErrors
  }

  private def buildParsedDependency(
      mainFileInfo: MainFileInfo,
      results: Seq[APBResult],
      scope: DependencyScope,
      migrationRisks: MigrationRisks
  ): Future[ParsedDependency] = {
    scope match {
      case ExtensionScope if mainFileInfo.baseUnit.isInstanceOf[Dialect] =>
        mainFileInfo.parseCompanionLibrary.map { companion =>
          companion.foreach(t => t._2.withReferences(Seq(mainFileInfo.baseUnit)))
          ExtensionDependency(mainFileInfo.baseUnit.asInstanceOf[Dialect], companion, mainFileInfo.descriptor, results)
        }
      case ValidationScope if mainFileInfo.baseUnit.isInstanceOf[DialectInstance] =>
        Future.successful(
          ProfileDependency(mainFileInfo.baseUnit.asInstanceOf[DialectInstance], mainFileInfo.descriptor, results)
        )
      case ManagementScope if mainFileInfo.baseUnit.isInstanceOf[JsonSchemaDocument] =>
        Future.successful(
          ManagementDependency(mainFileInfo.baseUnit.asInstanceOf[JsonSchemaDocument], mainFileInfo.descriptor, results)
        )
      case _ =>
        Future.successful(mainFileInfoToDependency(mainFileInfo, results, migrationRisks))
    }
  }

  private def mainFileInfoToDependency(
      mainFileInfo: MainFileInfo,
      results: Seq[APBResult],
      migrationRisks: MigrationRisks
  ) = {
    mainFileInfo match {
      case errorMainFileInfo: InvalidMainFileInfo =>
        InvalidParsedDependency(errorMainFileInfo.baseUnit, errorMainFileInfo.descriptor, results)
      case _ =>
        DesignDependency(
          mainFileInfo.baseUnit,
          mainFileInfo.descriptor,
          results,
          mainFileInfo.config.absolutePathListener,
          migrationRisks
        )
    }
  }

  protected def setDependencyGavAsLocation(
      dependency: ProjectDependency,
      errors: List[APBResult]
  ): List[APBResult] = {
    val location = DescriptorErrorLocation(dependency)
    errors.map(er => APBResult.forAspect(er.copy(location = Some(location)), er.aspect))
  }
}
