package org.mulesoft.apb.project.internal.engine

import amf.core.client.common.remote.Content
import amf.core.client.scala.resource.ResourceLoader
import amf.core.internal.unsafe.PlatformSecrets
import amf.custom.validation.internal.report.loaders.ProfileDialectLoader
import org.mulesoft.apb.project.client.scala.dependency._
import org.mulesoft.apb.project.client.scala.descriptor.{DescriptorParseResult, DescriptorParser}
import org.mulesoft.apb.project.client.scala.environment.{DependencyFetcher, UnreachableGavException}
import org.mulesoft.apb.project.client.scala.model.descriptor.{
  DependencyScope,
  Gav,
  ProjectDependency,
  ProjectDescriptor
}
import org.mulesoft.apb.project.client.scala.model.report.APBResult
import org.mulesoft.apb.project.client.scala.ProjectConfiguration
import org.mulesoft.apb.project.internal.dependency.{CyclicProjectDependencyException, UnreachableDependency}
import org.mulesoft.apb.project.internal.parser.APBEnv

import scala.collection.mutable
import scala.concurrent.ExecutionContext.Implicits.global
import scala.concurrent.Future
import scala.util.{Failure, Success}

abstract class TreeBuilderTemplate(
    dependencyFetcher: DependencyFetcher,
    loaders: List[ResourceLoader],
    cacheBuilder: UnitCacheBuilder
) extends ProjectEngine
    with PlatformSecrets {

  private type GavPath = String

  protected val cache: mutable.Map[GavPath, Future[Dependency]]  = mutable.Map.empty
  protected val gavLoaders: mutable.Map[GavPath, ResourceLoader] = mutable.Map.empty

  private val descriptorHandler: DescriptorParser = APBEnv.getHandler()

  protected def parseDescriptor(content: Content): DescriptorParseResult =
    descriptorHandler.parse(content.stream.toString)

  override def build(project: ProjectDescriptor): Future[ProjectConfiguration] =
    buildProjectConfig(project, Set.empty)

  protected def buildProjectConfig(
      descriptor: ProjectDescriptor,
      seenDependencies: Set[Gav]
  ): Future[ProjectConfiguration] = synchronized {
    // TODO: shouldn't have to filter, all dependencies should be processed here. Look at classifier to pick right config. What is the classifier of an extension?

    val (regularDependencies, managementDependencies) =
      descriptor.dependencies().partition(_.scope == DependencyScope.Management)

    val parsed = regularDependencies.map(dep => processDependency(descriptor, dep, seenDependencies)) ++
      managementDependencies.map(dep => processDependency(descriptor, dep, seenDependencies))

    val isRootProject = seenDependencies.isEmpty

    for {
      deps           <- Future.sequence(parsed)
      profileDialect <- ProfileDialectLoader.dialect
    } yield {
      val parsedDependencies                                = deps.collect({ case p: ParsedDependency => p })
      val (design, profile, extension, management, invalid) = PartitionedDependencies(parsedDependencies)
      var errors: Seq[APBResult]                            = dependencyErrors(deps)
      if (isRootProject) errors = errors ++ SuggestedExtensionAdditions(descriptor.gav(), parsedDependencies)
      val risks = deps.collect { case c: DesignDependency => c }.foldLeft(MigrationRisks.empty) { (acc, curr) =>
        acc.add(curr.risks)
      }

      ProjectConfiguration(
          design,
          profile,
          extension,
          management,
          invalid,
          gavLoaders.toMap,
          descriptor,
          profileDialect,
          cacheBuilder,
          loaders,
          errors,
          risks
      )
    }
  }

  private def dependencyErrors(seq: Seq[Dependency]) = seq.flatMap(_.results)

  private def assetIsUnreachable(dependency: ProjectDependency) = {
    !dependencyFetcher.accepts(dependency)
  }

  /** This method is responsible for checking cycles, checking if the dependency is in the cache and adding the Future
    * dependency to it. The method is synchronized to prevent race conditions to the cache.
    *
    * @param descriptor
    *   descriptor of the project that has this dependency
    * @param dependency
    *   dependency definitions as defined in the project descriptor
    * @param dependenciesInPath
    *   path of dependency from root project to current dependency
    *
    * @return
    *   a Future with the parsed dependency, this Future may came from the cache.
    */
  protected def processDependency(
      descriptor: ProjectDescriptor,
      dependency: ProjectDependency,
      dependenciesInPath: Set[Gav]
  ): Future[Dependency] = synchronized {
    if (dependenciesInPath.contains(dependency.gav))
      throw CyclicProjectDependencyException(dependency.gav, dependenciesInPath)
    else if (cache.contains(dependency.gav.path())) cache(dependency.gav.path())
    else if (assetIsUnreachable(dependency)) Future.successful(UnreachableDependency(descriptor, dependency))
    else {
      val loader = cachedFetchLoader(dependency)
      cached(dependency.gav) { () =>
        loader.fetch(APBEnv.projectProtocol + APBEnv.descriptorFileName).transformWith {
          case Success(content) =>
            parseDependency(dependency, content, dependenciesInPath, loader, descriptor.classifier())
          case Failure(exception) => onFetchFailure(descriptor, dependency, exception)
        }
      }
    }
  }

  private def onFetchFailure(descriptor: ProjectDescriptor, dependency: ProjectDependency, exception: Throwable) = {
    exception match {
      case UnreachableGavException(_) => Future.successful(UnreachableDependency(descriptor, dependency))
      case _                          => throw exception
    }
  }

  private def cached(gav: Gav)(
      runParse: () => Future[Dependency]
  ): Future[Dependency] = {
    val parsedDependency = runParse()
    cache.put(gav.path(), parsedDependency)
    parsedDependency
  }

  private def fetchLoader(dependency: ProjectDependency) = {
    dependencyFetcher.fetch(dependency)
  }

  protected def cachedFetchLoader(dependency: ProjectDependency) = {
    val loader = fetchLoader(dependency)
    gavLoaders.put(dependency.gav.path(), loader)
    loader
  }

  private def parseDependency(
      dependency: ProjectDependency,
      content: Content,
      dependenciesInPath: Set[Gav],
      loader: ResourceLoader,
      parentClassifier: Option[String]
  ): Future[Dependency] = {
    val parsed = parseDescriptor(content)
    parseDependency(parsed, dependency, dependenciesInPath, loader, parentClassifier)
  }

  protected def parseDependency(
      descriptor: DescriptorParseResult,
      dependency: ProjectDependency,
      dependenciesInPath: Set[Gav],
      loader: ResourceLoader,
      parentClassifier: Option[String]
  ): Future[Dependency]

  protected def errorLocation(gav: Gav) = s"exchange_modules/${gav.path}exchange.json"

  case class DependencyContext(cache: mutable.Map[GavPath, Future[Dependency]]) {}
}
