package org.mulesoft.apb.project.client.scala.model.project.management

import amf.shapes.client.scala.model.domain.jsonldinstance.JsonLDObject
import org.mulesoft.apb.project.client.scala.model.DynamicObject
import org.mulesoft.apb.project.client.scala.model.project.management.AlertFilter.equals
import org.mulesoft.apb.project.client.scala.model.project.management.SchemaIris.{
  API_REF,
  CONDITION,
  FILTERS,
  INTERVAL,
  MESSAGE,
  METRIC,
  NAME,
  NOTIFICATIONS,
  OPERATOR,
  POLICY_BINDING_REF,
  RECIPIENTS,
  RESOURCES,
  RESPONSE_CODES,
  RichString,
  SPEC,
  SUBJECT,
  THRESHOLD,
  TYPE
}
import org.mulesoft.apb.project.internal.model.GraphAccessors

case class Alert(override private[apb] val internal: JsonLDObject)
    extends GraphAccessors
    with DynamicObject
    with HasName
    with HasMetadata
    with HasAnnotations
    with HasSpec {

  def resources(): Seq[Resource] = specIfPresent.map(_.getObjectArray(RESOURCES).map(Resource(_))).getOrElse(Seq.empty)
  private[apb] def withResources(resources: Seq[Resource]) = update(SPEC) { spec =>
    spec.withProperty(RESOURCES, resources)
  }
  private[apb] def removeResources(): this.type = update(SPEC) { spec =>
    spec.removeProperty(RESOURCES)
  }

  private[apb] def targetsInstance(instanceName: String): Boolean = resources().exists(_.targetsInstance(instanceName))

  private[apb] def addResource(instance: APIInstance): this.type = {
    val containsResourceForInstance = resources().exists(_.targetsInstance(instance.name))
    if (!containsResourceForInstance) {
      withResources(resources() :+ Resource(instance.name, RESOURCES))
      this
    } else this
  }

  private[apb] def addResource(resource: Resource): this.type = {
    withResources(resources() :+ resource)
    this
  }

  private[apb] def updateResource(instance: APIInstance)(updater: Resource => Resource): this.type = {
    updateResource(res => res.apiRef().contains(instance.name))(updater)
  }

  private[apb] def updateResource(predicate: Resource => Boolean)(updater: Resource => Resource): this.type = {
    val next = resources().map { res =>
      if (predicate(res)) {
        updater(res)
      } else res
    }
    withResources(next)
    this
  }
  private[apb] def addFilter(instance: APIInstance, filter: AlertFilter) = {
    if (hasResourceFor(instance)) {
      updateResource(instance) { res => res.addFilter(filter) }
    } else addResource(Resource(instance.name, RESOURCES).withFilters(Seq(filter)))
  }

  private[apb] def hasResourceFor(instance: APIInstance) = {
    resources()
      .exists { res => res.targetsInstance(instance.name) }
  }

  def condition(): Option[AlertCondition] =
    specIfPresent.flatMap(_.getObjectIfPresent(CONDITION).map(AlertCondition(_)))
  def interval(): Option[Int] = specIfPresent.flatMap(_.getScalarIfPresent(INTERVAL))
  def notifications: Seq[AlertNotification] =
    specIfPresent.map(_.getObjectArray(NOTIFICATIONS).map(AlertNotification(_))).getOrElse(Seq.empty)
}

object Resource {
  def apply(apiRef: String, propertyIri: String): Resource = {
    val model = propertyIri.toModel
    val init = JsonLDObject
      .empty(model, model.path)
      .withProperty(API_REF, ApiRef(apiRef, API_REF))
    new Resource(init)
  }
}

case class Resource(override private[apb] val internal: JsonLDObject) extends GraphAccessors with DynamicObject {
  private[apb] def isEmpty: Boolean = !internal.fields.nonEmpty

  def apiRef(): Option[String]                         = getObjectIfPresent(API_REF).map(ApiRef(_)).map(_.name())
  private[apb] def withApiRef(name: String): this.type = withProperty(API_REF, ApiRef(name, API_REF))

  private[apb] def targetsInstance(instance: String): Boolean = apiRef().contains(instance)
  private[apb] def removeApiRef(): this.type = {
    this.removeProperty(API_REF)
    this
  }
  def filters(): Seq[AlertFilter] = getObjectArray(FILTERS).map(AlertFilter(_))
  private[apb] def addFilter(filter: AlertFilter): this.type = {
    val containsFilter = filters().exists(equals(_, filter))
    if (!containsFilter) {
      val next = filters() :+ filter
      withProperty(FILTERS, next)
    } else this
  }
  private[apb] def withFilters(filters: Seq[AlertFilter]): this.type = withProperty(FILTERS, filters)
  private[apb] def removeFilters(): this.type = {
    this.removeProperty(FILTERS)
    this
  }
}

object ApiRef {
  def apply(value: String, propertyIri: String): ApiRef = {
    val model = propertyIri.toModel
    val init = JsonLDObject
      .empty(model, model.path)
      .withProperty(NAME, value)
    new ApiRef(init)
  }
}

case class ApiRef(override private[apb] val internal: JsonLDObject) extends GraphAccessors with DynamicObject {

  def name(): String = getScalar[String](NAME)
}

case class AlertCondition(override private[apb] val internal: JsonLDObject) extends GraphAccessors with DynamicObject {

  def metric(): Option[String]    = getScalarIfPresent[String](METRIC)
  def operator(): Option[String]  = getScalarIfPresent[String](OPERATOR)
  def threshold(): Option[Double] = getScalarIfPresent[Double](THRESHOLD)
}

sealed trait AlertFilter extends GraphAccessors with DynamicObject

object AlertFilter {
  def apply(accessors: GraphAccessors): AlertFilter = {
    val isResponseCodesFilter = accessors.containsProperty(RESPONSE_CODES)
    if (isResponseCodesFilter) {
      ResponseCodesFilter(accessors)
    } else {
      PolicyBindingFilter(accessors)
    }
  }

  def equals(a: AlertFilter, b: AlertFilter) = (a, b) match {
    case (a: ResponseCodesFilter, b: ResponseCodesFilter) => a == b
    case (a: PolicyBindingFilter, b: PolicyBindingFilter) => a == b
    case _                                                => false
  }
}

case class ResponseCodesFilter(override private[apb] val internal: JsonLDObject) extends AlertFilter {
  def values(): Seq[Int] = getScalarArray[Int](RESPONSE_CODES)
}

object PolicyBindingFilter {
  def apply(name: String, propertyIri: String): PolicyBindingFilter = {
    val model = propertyIri.toModel
    val ref   = PolicyBindingRef(name, POLICY_BINDING_REF)
    val init = JsonLDObject
      .empty(model, model.path)
      .withProperty(POLICY_BINDING_REF, ref)
    new PolicyBindingFilter(init)
  }
}

case class PolicyBindingFilter(override private[apb] val internal: JsonLDObject) extends AlertFilter {
  def ref(): PolicyBindingRef = new PolicyBindingRef(getObject(POLICY_BINDING_REF))
}

object PolicyBindingRef {
  def apply(value: String, propertyIri: String): PolicyBindingRef = {
    val model = propertyIri.toModel
    val init = JsonLDObject
      .empty(model, model.path)
      .withProperty(NAME, value)
    new PolicyBindingRef(init)
  }
}
case class PolicyBindingRef(override private[apb] val internal: JsonLDObject)
    extends GraphAccessors
    with DynamicObject {

  def name(): String = getScalar[String](NAME)
}

case class AlertNotification(override private[apb] val internal: JsonLDObject)
    extends GraphAccessors
    with DynamicObject {

  def message(): Option[String] = getScalarIfPresent[String](MESSAGE)
  def subject(): Option[String] = getScalarIfPresent[String](SUBJECT)
  def recipients(): Seq[String] = getScalarArray[String](RECIPIENTS)
  def `type`(): Option[String]  = getScalarIfPresent[String](TYPE)
}
