package org.mulesoft.apb.project.client.scala.config

import amf.apb.internal.plugins.ApbFallbackPlugin
import amf.apicontract.client.scala.{AMFConfiguration, APIConfiguration}
import amf.apicontract.internal.transformation.UnsupportedTransformationPipeline
import amf.core.client.scala.config.UnitCache
import amf.core.client.scala.resource.ResourceLoader
import amf.core.internal.remote.Spec
import org.mulesoft.apb.project.client.scala.{DependencySet, ProjectConfiguration}
import org.mulesoft.apb.project.client.scala.dependency.{CloneOnceUnitCacheBuilder, UnitCacheBuilder}
import org.mulesoft.apb.project.client.scala.model.descriptor.ProjectDescriptor
import org.mulesoft.apb.project.internal.dependency.APBResourceLoader
import org.mulesoft.apb.project.internal.dependency.config.ConfigProvider
import org.mulesoft.apb.project.internal.idadoption.IdAdopterProvider
import org.mulesoft.apb.project.internal.listener.{
  AbsolutePathDetectionRawReferenceListener,
  InternalDependencyFileReferenceListener
}
import org.mulesoft.apb.project.internal.transformation.APBTransformationPipeline

object AMFConfigurationFactory {

  def createFromClassifier(descriptor: ProjectDescriptor): AMFConfiguration = {
    descriptor
      .classifier()
      .map(ConfigProvider.fromClassifier)
      .getOrElse(APIConfiguration.APIWithJsonSchema())
      .withFallback(ApbFallbackPlugin())
  }

  def createFrom(
      config: DependencySet,
      base: AMFConfiguration,
      loaders: Seq[ResourceLoader],
      cacheBuilder: UnitCacheBuilder = CloneOnceUnitCacheBuilder
  ): AMFConfiguration =
    dependenciesConfig(config, base, loaders, cacheBuilder)

  def createSpecAgnosticParsingConfig(
      config: DependencySet,
      loaders: Seq[ResourceLoader],
      cacheBuilder: UnitCacheBuilder = CloneOnceUnitCacheBuilder
  ): AMFConfiguration = {
    dependenciesConfig(config, APIConfiguration.APIWithJsonSchema(), loaders, cacheBuilder)
  }

  def createSpecificationParsingConfig(
      config: DependencySet,
      loaders: Seq[ResourceLoader],
      cacheBuilder: UnitCacheBuilder = CloneOnceUnitCacheBuilder
  ): AMFConfiguration = {
    dependenciesConfig(config, createFromClassifier(config.descriptor()), loaders, cacheBuilder)
  }

  private[apb] def createFromSpec(
      config: DependencySet,
      spec: Spec,
      loaders: List[ResourceLoader],
      cacheBuilder: UnitCacheBuilder = CloneOnceUnitCacheBuilder
  ): AMFConfiguration = {
    dependenciesConfig(config, ConfigProvider.fromSpec(spec), loaders, cacheBuilder)
  }

  private def dependenciesConfig(
      config: DependencySet,
      base: AMFConfiguration,
      loaders: Seq[ResourceLoader],
      cacheBuilder: UnitCacheBuilder
  ): AMFConfiguration = {
    base
      .withResourceLoaders(List(mainLoader(config)) ++ loaders)
      .withDialect(config.validationDialect())
      .withUnitCache(buildUnitCache(config, cacheBuilder))
      .addListeners(config)
      .addExtensions(config)
      .withIdAdopterProvider(IdAdopterProvider)
      .withPipelines(config)
  }

  def buildUnitCache(config: DependencySet, cacheBuilder: UnitCacheBuilder): UnitCache = {
    val designDependencies             = config.design()
    val extensionCompanionDependencies = config.designExtension().flatMap(_.companionLibDependency())
    cacheBuilder.build(designDependencies ++ extensionCompanionDependencies)
  }

  implicit class AMFConfigPopulator(base: AMFConfiguration) {
    def addExtensions(config: DependencySet): AMFConfiguration =
      config.designExtension().foldLeft(base) { (acc, curr) =>
        acc.withDialect(curr.dialect)
      }

    def addListeners(config: DependencySet): AMFConfiguration = {
      val listeners = config match {
        case legacyConfig: ProjectConfiguration => legacyConfig.getListeners()
        case _ =>
          AbsolutePathDetectionRawReferenceListener(config.descriptor()) ++ Seq(
              InternalDependencyFileReferenceListener(config.design())
          )
      }
      listeners.foldLeft(base) { (acc, curr) =>
        acc.withEventListener(curr)
      }
    }

    def withPipelines(config: DependencySet): AMFConfiguration = overwritePipelines(config, base)
  }

  private def overwritePipelines(config: DependencySet, base: AMFConfiguration) = {
    val newPipelines = base.configurationState().getTransformationPipelines().flatMap {
      case unsupported: UnsupportedTransformationPipeline => Option(unsupported)
      case pipeline                                       => Option(APBTransformationPipeline(config, pipeline))
    }
    newPipelines.foldLeft(base) { (config, pipeline) => config.withTransformationPipeline(pipeline) }
  }

  private def mainLoader(dependencies: DependencySet): ResourceLoader = new APBResourceLoader(
      dependencies.dependencyLoaders()
  )
}
