package org.mulesoft.apb.project.client.scala.model.descriptor

import amf.core.client.scala.model.{BoolField, StrField}
import amf.core.internal.parser.domain.Fields
import amf.shapes.client.scala.model.domain.jsonldinstance.JsonLDObject
import amf.shapes.internal.spec.jsonldschema.parser.JsonPath
import org.mulesoft.apb.project.client.scala.model.DynamicObject
import org.mulesoft.apb.project.client.scala.model.descriptor.documentation.Documentation
import org.mulesoft.apb.project.client.scala.model.descriptor.community.Community
import org.mulesoft.apb.project.client.scala.model.descriptor.governance.GovernanceInfo
import org.mulesoft.apb.project.internal.generated.DescriptorSchemaLoader.LegacyVersion
import org.mulesoft.apb.project.internal.model.ProjectDescriptorModel.{Metadata => MetadataIri, _}
import org.mulesoft.apb.project.internal.model.descriptor.ProjectDependencyModel.{AssetId, Classifier, GroupId, Version}
import org.mulesoft.apb.project.internal.model.{GraphAccessors, ProjectBase, ProjectDescriptorModel}
import org.mulesoft.common.time.SimpleDateTime

import scala.collection.mutable
import scala.language.implicitConversions

case class ProjectDescriptor private[apb] (override private[apb] val internal: JsonLDObject)
    extends DynamicObject
    with GraphAccessors
    with ProjectBase {

  override type This = ProjectDescriptor
  private implicit def metadataConversion(obj: JsonLDObject): Option[Metadata]    = Option(obj).map(Metadata.apply)
  private implicit def instanceConversion(objs: Seq[JsonLDObject]): Seq[Instance] = objs.map(Instance.apply)
  private implicit def documentationConversion(objs: Seq[JsonLDObject]): Seq[Documentation] =
    objs.map(Documentation.apply)
  private implicit def communityConversion(objs: Seq[JsonLDObject]): Seq[Community] = objs.map(Community.apply)
  private implicit def governanceInfoConversion(obj: JsonLDObject): Option[GovernanceInfo] =
    Option(obj).map(GovernanceInfo.apply)

  def main: Option[String]                   = get[StrField](ProjectDescriptorModel.Main)
  def descriptorVersion: String              = getDescriptorVersion
  def instances: Seq[Instance]               = get[Seq[JsonLDObject]](ApiInstances)
  def originalFormatVersion: Option[String]  = get[StrField](OriginalFormatVersion)
  def metadata: Option[Metadata]             = get[JsonLDObject](MetadataIri)
  def backwardsCompatible: Option[Boolean]   = get[BoolField](BackwardsCompatible)
  def publishWithRefFiles: Option[Boolean]   = get[BoolField](PublishWithRefFiles)
  def projectId: Option[String]              = get[StrField](ProjectId)
  def documentation: Seq[Documentation]      = get[Seq[JsonLDObject]](ProjectDescriptorModel.Documentation)
  def communities: Seq[Community]            = get[Seq[JsonLDObject]](ProjectDescriptorModel.Communities)
  def governanceInfo: Option[GovernanceInfo] = get[JsonLDObject](ProjectDescriptorModel.GovernanceInfo)
  override protected def fields: Fields      = internal.fields

  private def getDescriptorVersion: String = {
    val version: StrField = get(DescriptorVersion)
    version.option() match {
      case Some("1.0.0")       => "1.0.0"
      case Some(LegacyVersion) => LegacyVersion
      case Some(_)             => "1.0.0"
      case None                => LegacyVersion
    }
  }

  private[apb] def isRaml: Boolean = classifier().contains("raml")

  def isLegacyDescriptor: Boolean = descriptorVersion == LegacyVersion

  def withMain(main: String): ProjectDescriptor = {
    internal.withProperty(ProjectDescriptorModel.Main, main)
    this
  }

  def withDependency(dependency: ProjectDependency): ProjectDescriptor = {
    val newDependencies = dependencies().filterNot(_.gav == dependency.gav) :+ dependency
    internal.withObjPropertyCollection(Dependencies, newDependencies.map(_.internal))
    this
  }

  def withDependencies(dependencies: Seq[ProjectDependency]): ProjectDescriptor = {
    internal.withObjPropertyCollection(Dependencies, dependencies.map(_.internal))
    this
  }

  def withoutDependency(dependency: ProjectDependency): ProjectDescriptor = {
    val newDependencies = dependencies().filterNot(_.gav == dependency.gav)
    internal.withObjPropertyCollection(Dependencies, newDependencies.map(_.internal))
    this
  }

  def withInstance(instance: Instance): ProjectDescriptor = {
    val newInstances = instances.filterNot(_.gcl == instance.gcl) :+ instance
    withInstances(newInstances)
  }

  def withInstances(instances: Seq[Instance]): ProjectDescriptor = {
    internal.withObjPropertyCollection(ApiInstances, instances.map(_.internal))
    this
  }

  def withName(name: String): ProjectDescriptor = {
    internal.withProperty(Name, name)
    this
  }

  def withDescriptorVersion(version: String): ProjectDescriptor = {
    internal.withProperty(DescriptorVersion, version)
    this
  }

  def withGav(gav: Gav): ProjectDescriptor = {
    internal
      .withProperty(GroupId, gav.groupId)
      .withProperty(AssetId, gav.assetId)
    withVersion(gav.version)
    this
  }

  def withVersion(version: String): ProjectDescriptor = {
    internal.withProperty(Version, version)
    this
  }

  def withClassifier(classifier: String): ProjectDescriptor = {
    internal.withProperty(Classifier, classifier)
    this
  }

  def withTags(tags: Seq[String]): ProjectDescriptor = {
    internal.withStringPropertyCollection(Tags, tags)
    this
  }

  def withTag(tag: String): ProjectDescriptor = withTags(tags :+ tag)

  def withApiVersion(apiVersion: String): ProjectDescriptor = {
    internal.withProperty(ApiVersion, apiVersion)
    this
  }

  def withOrganizationId(organizationId: String): ProjectDescriptor = {
    internal.withProperty(OrganizationId, organizationId)
    this
  }

  def withOriginalFormatVersion(originalFormatVersion: String): ProjectDescriptor = {
    internal.withProperty(OriginalFormatVersion, originalFormatVersion)
    this
  }

  def withMetadata(metadata: Metadata): ProjectDescriptor = {
    internal.withProperty(MetadataIri, metadata.internal)
    this
  }

  def withBackwardsCompatible(value: Boolean): ProjectDescriptor = {
    internal.withProperty(BackwardsCompatible, value)
    this
  }

  def withPublishWithRefFiles(value: Boolean): ProjectDescriptor = {
    internal.withProperty(PublishWithRefFiles, value)
    this
  }

  def withDescription(description: String): ProjectDescriptor = {
    internal.withProperty(Description, description)
    this
  }

  def withProjectId(projectId: String): ProjectDescriptor = {
    internal.withProperty(ProjectId, projectId)
    this
  }

  def withStatus(status: String): ProjectDescriptor = {
    internal.withProperty(Status, status)
    this
  }

  def withCreatedAt(createdAt: SimpleDateTime): ProjectDescriptor = {
    internal.withDateTimeProperty(CreatedAt, createdAt)
    this
  }

  def withCreatedBy(createdBy: String): ProjectDescriptor = {
    internal.withProperty(CreatedBy, createdBy)
    this
  }

  def withUpdatedAt(updatedAt: SimpleDateTime): ProjectDescriptor = {
    internal.withDateTimeProperty(UpdatedAt, updatedAt)
    this
  }

  def withSystemUpdateTime(systemUpdateTime: SimpleDateTime): ProjectDescriptor = {
    internal.withDateTimeProperty(SystemUpdatedAt, systemUpdateTime)
    this
  }

  def withTenantId(tenantId: String): ProjectDescriptor = {
    internal.withProperty(TenantId, tenantId)
    this
  }

  def withCategories(categories: Seq[String]): ProjectDescriptor = {
    internal.withStringPropertyCollection(Categories, categories)
    this
  }

  def withFields(fields: Seq[String]): ProjectDescriptor = {
    internal.withStringPropertyCollection(CustomFields, fields)
    this
  }

  def withKeyValueTags(keyValueTags: Seq[String]): ProjectDescriptor = {
    internal.withStringPropertyCollection(KeyValueTags, keyValueTags)
    this
  }

  def withType(`type`: String): ProjectDescriptor = {
    internal.withProperty(TypeField, `type`)
    this
  }

  def withMajorVersionComponent(version: Int): ProjectDescriptor = {
    internal.withProperty(MajorVersionComponent, version)
    this
  }

  def withMinorVersionComponent(version: Int): ProjectDescriptor = {
    internal.withProperty(MinorVersionComponent, version)
    this
  }

  def withPatchVersionComponent(version: Int): ProjectDescriptor = {
    internal.withProperty(PatchVersionComponent, version)
    this
  }

  def withApiId(apiId: String): ProjectDescriptor = {
    internal.withProperty(ApiId, apiId)
    this
  }

  def withProjectType(projectType: String): ProjectDescriptor = {
    internal.withProperty(ProjectType, projectType)
    this
  }

  def withDocumentation(doc: Documentation): ProjectDescriptor = {
    val newInstances = documentation.filterNot(_.name.contains(doc.name.getOrElse(""))) :+ doc
    withDocumentation(newInstances)
  }

  def withDocumentation(docs: Seq[Documentation]): ProjectDescriptor = {
    internal.withObjPropertyCollection(ProjectDescriptorModel.Documentation, docs.map(_.internal))
    this
  }

  def withCommunity(doc: Community): ProjectDescriptor = {
    withCommunities(communities :+ doc)
  }

  def withCommunities(docs: Seq[Community]): ProjectDescriptor = {
    internal.withObjPropertyCollection(ProjectDescriptorModel.Communities, docs.map(_.internal))
    this
  }

  def withContactName(contactName: String): ProjectDescriptor = {
    internal.withProperty(ContactName, contactName)
    this
  }

  def withContactEMail(contactEMail: String): ProjectDescriptor = {
    internal.withProperty(ContactEMail, contactEMail)
    this
  }

  def withProjectETag(projectETag: String): ProjectDescriptor = {
    internal.withProperty(ProjectETag, projectETag)
    this
  }

  def withGovernanceInfo(governanceInfo: GovernanceInfo): ProjectDescriptor = {
    internal.withProperty(ProjectDescriptorModel.GovernanceInfo, governanceInfo.internal)
    this
  }

  def withHasDocumentation(hasDocumentation: Boolean): ProjectDescriptor = {
    internal.withProperty(HasDocumentation, hasDocumentation)
    this
  }

  def withApiUrn(apiUrn: String): ProjectDescriptor = {
    internal.withProperty(ApiUrn, apiUrn)
    this
  }

  def withAttributes(attributes: Seq[AttributeEntry]): ProjectDescriptor = {
    internal.withObjPropertyCollection(Attribute, attributes.map(_.internal))
    this
  }

  def withManagedTags(managedTags: Seq[String]): ProjectDescriptor = {
    internal.withStringPropertyCollection(ManagedTag, managedTags)
    this
  }

  def withFilePackaging(filePackaging: Seq[FilePackaging]): ProjectDescriptor = {
    internal.withObjPropertyCollection(ProjectDescriptorModel.FilePackaging, filePackaging.map(_.internal))
    this
  }

  def copy() = new ProjectDescriptor(internal.cloneElement(mutable.Map.empty).asInstanceOf[JsonLDObject])
}

object ProjectDescriptor {
  def apply(main: String): ProjectDescriptor = {
    val inner = createInner().set(Main, main)
    new ProjectDescriptor(inner)
  }

  def apply(): ProjectDescriptor = new ProjectDescriptor(createInner())

  private def createInner() = {
    JsonLDObject.empty(ProjectDescriptorModel.entityModel, JsonPath.empty)
  }
}
