package org.mulesoft.apb.project.internal.instances

import org.mulesoft.apb.project.client.scala.model.project.management.{
  APIInstance,
  Alert,
  PolicyBinding,
  PolicyBindingFilter,
  Resource
}
import org.mulesoft.apb.project.internal.instances.AlertIndex.{ApiInstanceName, PolicyBindingName}

import scala.collection.mutable

object AlertIndex {

  type ApiInstanceName   = String
  type PolicyBindingName = String

  sealed trait AlertTarget

  case class TargetsPolicy(instance: String, binding: String, alert: Alert) extends AlertTarget
  case class TargetsInstance(instance: String, alert: Alert)                extends AlertTarget

  def build(catalog: ResourceCatalog): AlertIndex = {
    val targets              = catalog.getAlerts.flatMap(getAlertTarget)
    val forApiInstanceAcc    = mutable.Map.empty[String, Seq[Alert]]
    val forBindingInInstance = mutable.Map.empty[(ApiInstanceName, PolicyBindingName), Seq[Alert]]

    targets.foreach {
      case TargetsInstance(instance, alert)       => update(forApiInstanceAcc, instance, alert)
      case TargetsPolicy(instance, policy, alert) => update(forBindingInInstance, (instance, policy), alert)
    }

    AlertIndex(forApiInstanceAcc.toMap, forBindingInInstance.toMap)
  }

  private def update[K, V](map: mutable.Map[K, Seq[V]], key: K, value: V): mutable.Map[K, Seq[V]] = {
    val newValue = map.get(key).map(seq => seq :+ value).getOrElse(Seq(value))
    map.put(key, newValue)
    map
  }

  private def getAlertTarget(alert: Alert): Seq[AlertTarget] = {
    val targets = alert.resources().flatMap { res =>
      res.apiRef() match {
        case Some(apiRef) =>
          val targets = findPolicyBindingTargets(res)
          if (targets.isEmpty) List(TargetsInstance(apiRef, alert))
          else targets.map(target => TargetsPolicy(apiRef, target, alert))
        case None => List.empty
      }
    }
    targets
  }

  private def findPolicyBindingTargets(resource: Resource) = resource
    .filters()
    .collect { case policyFilter: PolicyBindingFilter =>
      policyFilter.ref()
    }
    .map(_.name())
}

case class AlertIndex(
    private val forApiInstance: Map[String, Seq[Alert]] = Map.empty,
    private val forBindingInInstance: Map[(ApiInstanceName, PolicyBindingName), Seq[Alert]] = Map.empty
) {

  def findForPolicyBinding(instance: APIInstance, binding: PolicyBinding): Seq[Alert] = {
    val key = (instance.name, binding.name)
    forBindingInInstance.getOrElse(key, Seq.empty)
  }

  def findForPolicyBinding(instance: APIInstance): Seq[Alert] =
    instance.policyBindings.flatMap(findForPolicyBinding(instance, _))

  def findForApiInstance(instance: APIInstance): Seq[Alert] = forApiInstance.getOrElse(instance.name, Seq.empty)
}
