package org.mulesoft.apb.project.internal.transformation.steps

import amf.core.client.scala.AMFGraphConfiguration
import amf.core.client.scala.errorhandling.AMFErrorHandler
import amf.core.client.scala.model.document.BaseUnit
import amf.core.client.scala.model.domain.{AmfElement, AmfObject}
import amf.core.client.scala.transform.TransformationStep
import amf.core.internal.annotations.ResolvedReferenceLike
import amf.shapes.client.scala.model.document.{JsonLDInstanceDocument, JsonSchemaDocument}
import amf.shapes.client.scala.model.domain.jsonldinstance.JsonLDObject
import org.mulesoft.apb.project.client.scala.DependencySet
import org.mulesoft.apb.project.client.scala.model.descriptor.Gav
import org.mulesoft.apb.project.internal.common.Classifier
import org.mulesoft.apb.project.internal.transformation.steps.AgentNetworkMetadata._
import org.mulesoft.apb.project.internal.validations.AgentNetworkValidations.InvalidReferenceLike

import scala.collection.mutable

class ReferenceLikeResolutionStage(dependencySet: DependencySet) extends TransformationStep() {

  override def transform(
      model: BaseUnit,
      errorHandler: AMFErrorHandler,
      configuration: AMFGraphConfiguration
  ): BaseUnit = {
    model match {
      case jsonDoc: JsonLDInstanceDocument if applies =>
        val refLikeNodes: mutable.Set[ReferenceLike]           = mutable.Set.empty
        val declarationLikeNodes: mutable.Set[DeclarationLike] = mutable.Set.empty
        val dependencyMap: Map[String, DependencyInformation]  = generateDependencyMap
        jsonDoc
          .iterator()
          .toStream
          .foreach(e => collectReferenceAndDeclarationsLike(e, refLikeNodes, declarationLikeNodes))

        validateAndResolveReferences(refLikeNodes, declarationLikeNodes, dependencyMap, errorHandler)
    }
    model
  }

  private def applies: Boolean = dependencySet.descriptor().classifier().contains(Classifier.AGENT_NETWORK)

  private def generateDependencyMap: Map[String, DependencyInformation] = dependencySet
    .allDependencies()
    .map { dependency =>
      (
        minimalGav(dependency.descriptor.gav()),
        DependencyInformation(dependency.descriptor.classifier(), dependency.baseUnit)
      )
    }
    .toMap

  private def minimalGav(gav: Gav): String = s"${gav.groupId}/${gav.assetId}"

  private def collectReferenceAndDeclarationsLike(
      element: AmfElement,
      references: mutable.Set[ReferenceLike],
      declarations: mutable.Set[DeclarationLike]
  ): Unit = {
    element match {
      case obj: JsonLDObject =>
        obj.typeIris.headOption match {
          case Some(DECLARATIONS) => collectDeclarationsLike(obj, declarations)
          case Some(REFERENCES)   => collectReferenceLike(obj, references)
          case _                  => // Ignore
        }
    }
  }

  private def collectDeclarationsLike(
      declarationsMap: JsonLDObject,
      declarations: mutable.Set[DeclarationLike]
  ): Unit = {
    val kind = getReferenceKind(declarationsMap)
    declarationsMap.fields.foreach { case (_, value) =>
      value.value match {
        case declarationEntry: JsonLDObject => collectDeclarationLike(declarationEntry, declarations, kind)
        case _                              => // Ignore, this will be handled by validation
      }
    }
  }

  private def collectDeclarationLike(
      obj: JsonLDObject,
      declarations: mutable.Set[DeclarationLike],
      kind: String
  ): Unit = {
    val declaration = DeclarationLike(getKey(obj), obj, kind)
    declarations.add(declaration)
  }

  private def collectReferenceLike(obj: JsonLDObject, references: mutable.Set[ReferenceLike]): Unit = {
    val kind      = getReferenceKind(obj)
    val reference = ReferenceLike(getName(obj), getNamespace(obj), obj, kind)
    references.add(reference)
  }

  private def getReferenceKind(obj: JsonLDObject): String = {
    obj.typeIris.find(iri => iri.startsWith(KIND_BASE)).map(_.stripPrefix(KIND_BASE)).getOrElse("")
  }

  private def getKey(obj: JsonLDObject): String = obj.path.lastSegment.getOrElse("")

  private def getName(obj: JsonLDObject): String =
    obj.fields.getValueAsOption(NAME).map(_.value.toString).getOrElse("")

  private def getNamespace(obj: JsonLDObject): Option[String] =
    obj.fields.getValueAsOption(NAMESPACE).map(_.value.toString)

  private def validateAndResolveReferences(
      references: mutable.Set[ReferenceLike],
      declarations: mutable.Set[DeclarationLike],
      dependencies: Map[String, DependencyInformation],
      eh: AMFErrorHandler
  ): Unit = {
    references.foreach { reference =>
      if (!localReference(reference, declarations) && !dependencyReference(reference, dependencies)) {
        reference.value.annotations += ResolvedReferenceLike.unresolved()
        if (reference.kind != CONNECTION_SUFFIX) // Connection is validated externally. Don't want to generate an error
          reportReferenceError(reference, eh)
      }
    }
  }

  private def localReference(ref: ReferenceLike, declarations: mutable.Set[DeclarationLike]): Boolean = {
    val applies = declarations.filter(declaration => ref.target == declaration.name && ref.kind == declaration.kind)
    applies.headOption.foreach(declaration => addResolvedAnnotation(ref.value, declaration.value))
    applies.nonEmpty
  }

  private def dependencyReference(ref: ReferenceLike, dependencies: Map[String, DependencyInformation]): Boolean = {
    val targetDependency         = s"${dependencyNamespace(ref)}/${ref.target}"
    val allowedTargetClassifiers = refKindToClassifier(ref.kind)
    val applies = dependencies.get(targetDependency).filter(d => d.classifier.exists(allowedTargetClassifiers.contains))
    applies.foreach(dependency => addResolvedAnnotation(ref.value, dependency.baseUnit))
    applies.nonEmpty
  }

  // The dependency groupId is set in the reference as namespace or is the same groupId as the agent-network asset
  private def dependencyNamespace(ref: ReferenceLike): String =
    ref.namespace.getOrElse(dependencySet.descriptor().gav().groupId)

  private def reportReferenceError(ref: ReferenceLike, eh: AMFErrorHandler): Unit = {
    eh.violation(
      InvalidReferenceLike,
      ref.value,
      s"Cannot resolve reference ${ref.target} of kind ${ref.kind}",
      ref.value.annotations
    )
  }

  // TODO currently this is only adding an annotation with the result, not the content itself
  private def addResolvedAnnotation(refNode: JsonLDObject, targetNode: AmfObject): Unit = {
    targetNode match {
      case obj: JsonLDObject => // Case for all local declarations
        refNode.annotations += ResolvedReferenceLike.local()
      case jsonDoc: JsonLDInstanceDocument => // Case for almost all remote assets
        refNode.annotations += ResolvedReferenceLike.remote()
      case schemaDoc: JsonSchemaDocument => // Case for POLICY_DEFINITION remote asset
        refNode.annotations += ResolvedReferenceLike.remote()
      case _ => // Ignored
    }
  }

  private case class DependencyInformation(classifier: Option[String], baseUnit: BaseUnit)

  private case class DeclarationLike(name: String, value: JsonLDObject, kind: String)
  private case class ReferenceLike(target: String, namespace: Option[String], value: JsonLDObject, kind: String)
}
