package org.mulesoft.apb.project.internal.dependency.config

import amf.agentcard.client.scala.AgentCardConfiguration
import amf.agentmetadata.client.scala.AgentMetadataConfiguration
import amf.agentnetwork.client.scala.AgentNetworkConfiguration
import amf.apicontract.client.scala._
import amf.core.internal.remote.Spec
import amf.graphql.client.scala.GraphQLConfiguration
import amf.graphqlfederation.client.scala.GraphQLFederationConfiguration
import amf.grpc.client.scala.GRPCConfiguration
import amf.llmmetadata.client.scala.LLMMetadataConfiguration
import amf.mcp.client.scala.MCPConfiguration
import amf.othercard.client.scala.OtherCardConfiguration
import amf.shapes.client.scala.config.JsonSchemaConfiguration
import org.mulesoft.apb.project.internal.common.Classifier

object ConfigProvider {

  def fromClassifier(classifier: String): AMFConfiguration = classifier match {
    case Classifier.RAML | Classifier.OAS | Classifier.ASYNC => APIConfiguration.API()
    case Classifier.GRAPHQL                                  => GraphQLConfiguration.GraphQL()
    case Classifier.GRAPHQL_APOLLO                           => GraphQLFederationConfiguration.GraphQLFederation()
    case Classifier.GRPC                                     => GRPCConfiguration.GRPC()
    case Classifier.GRPC_LIBRARY                             => GRPCConfiguration.GRPC()
    case Classifier.JSON_SCHEMA    => ConfigurationAdapter.adapt(JsonSchemaConfiguration.JsonSchema())
    case Classifier.OAS_COMPONENT  => OASConfiguration.OASComponent()
    case Classifier.AVRO_SCHEMA    => AvroConfiguration.Avro()
    case Classifier.MCP            => ConfigurationAdapter.adapt(MCPConfiguration.MCP())
    case Classifier.AGENT_NETWORK  => ConfigurationAdapter.adapt(AgentNetworkConfiguration.AgentNetwork())
    case Classifier.AGENT_CARD     => ConfigurationAdapter.adapt(AgentCardConfiguration.AgentCard())
    case Classifier.OTHER_CARD     => ConfigurationAdapter.adapt(OtherCardConfiguration.OtherCard())
    case Classifier.AGENT_METADATA => ConfigurationAdapter.adapt(AgentMetadataConfiguration.AgentMetadata())
    case Classifier.LLM_METADATA   => ConfigurationAdapter.adapt(LLMMetadataConfiguration.LLMMetadata())
    case Classifier.POLICY_SCHEMA  => ConfigurationAdapter.adapt(JsonSchemaConfiguration.JsonSchema())
    case _                         => APIConfiguration.API()
  }

  def fromSpec(spec: Spec): AMFConfiguration = spec match {
    case Spec.RAML08 => RAMLConfiguration.RAML08()
    case Spec.RAML10 => RAMLConfiguration.RAML10()
    case Spec.OAS20  => OASConfiguration.OAS20()
    case Spec.OAS30  => OASConfiguration.OAS30()
    case Spec.OAS31  => OASConfiguration.OAS31()
    case Spec.ASYNC20 | Spec.ASYNC21 | Spec.ASYNC22 | Spec.ASYNC23 | Spec.ASYNC24 | Spec.ASYNC25 | Spec.ASYNC26 =>
      AsyncAPIConfiguration.Async20()
    case Spec.JSONSCHEMA         => ConfigurationAdapter.adapt(JsonSchemaConfiguration.JsonSchema())
    case Spec.GRAPHQL            => GraphQLConfiguration.GraphQL()
    case Spec.GRAPHQL_FEDERATION => GraphQLFederationConfiguration.GraphQLFederation()
    case Spec.GRPC               => GRPCConfiguration.GRPC()
    case Spec.AVRO_SCHEMA        => AvroConfiguration.Avro()
    case Spec.MCP                => ConfigurationAdapter.adapt(MCPConfiguration.MCP())
    case Spec.AGENT_NETWORK      => ConfigurationAdapter.adapt(AgentNetworkConfiguration.AgentNetwork())
    case Spec.AGENT_CARD         => ConfigurationAdapter.adapt(AgentCardConfiguration.AgentCard())
    case Spec.OTHER_CARD         => ConfigurationAdapter.adapt(OtherCardConfiguration.OtherCard())
    case Spec.AGENT_METADATA     => ConfigurationAdapter.adapt(AgentMetadataConfiguration.AgentMetadata())
    case Spec.LLM_METADATA       => ConfigurationAdapter.adapt(LLMMetadataConfiguration.LLMMetadata())
    case _ =>
      throw UnrecognizedSpecException(
        s"Spec ${spec.id} not supported. Supported specs are ${Spec.RAML08.id}, ${Spec.RAML10.id}, ${Spec.OAS20.id}, ${Spec.OAS30.id}, ${Spec.ASYNC20.id}, ${Spec.JSONSCHEMA.id}, ${Spec.GRAPHQL.id}, ${Spec.GRAPHQL_FEDERATION.id}, ${Spec.GRPC.id}, ${Spec.AVRO_SCHEMA.id}, ${Spec.MCP.id}, ${Spec.AGENT_NETWORK.id}, ${Spec.AGENT_CARD.id}, ${Spec.AGENT_CARD.id}, ${Spec.OTHER_CARD.id}, ${Spec.AGENT_METADATA.id}, ${Spec.LLM_METADATA.id}"
      )
  }
}
