package org.mulesoft.apb.project.internal.engine

import amf.core.client.common.validation.SeverityLevels
import amf.core.client.scala.resource.ResourceLoader
import org.mulesoft.apb.project.client.scala.dependency.{Dependency, IgnoredDependency}
import org.mulesoft.apb.project.client.scala.descriptor.DescriptorParseResult
import org.mulesoft.apb.project.client.scala.environment.DependencyFetcher
import org.mulesoft.apb.project.client.scala.model.descriptor.{DesignScope, Gav, ProjectDependency, ProjectDescriptor}
import org.mulesoft.apb.project.client.scala.model.report.{APBResult, ProjectAspects}
import org.mulesoft.apb.project.internal.dependency.NothingUnitCacheBuilder
import org.mulesoft.apb.project.internal.validations.ProjectValidations.ScopePresentInLegacyProject

import scala.concurrent.ExecutionContext.Implicits.global
import scala.concurrent.Future

case class LegacyTreeBuilder(dependencyFetcher: DependencyFetcher, loaders: List[ResourceLoader])
    extends TreeBuilderTemplate(dependencyFetcher, loaders, NothingUnitCacheBuilder) {

  override protected def parseDependency(
      descriptor: DescriptorParseResult,
      dependency: ProjectDependency,
      dependenciesInPath: Set[Gav],
      loader: ResourceLoader,
      parentClassifier: Option[String]
  ): Future[Dependency] = {
    val DescriptorParseResult(project, results) = descriptor
    val processedDependencies = project
      .dependencies()
      .map(dep => processDependency(project, dep, dependenciesInPath + project.gav))

    // we don't care about the dependencies, we only care the get processed so we can use them in resource loader
    Future.sequence(processedDependencies).map { _ =>
      val isRoot = dependenciesInPath.isEmpty
      val errors = scopeErrors(project, dependency, results, isRoot = isRoot)
      IgnoredDependency(project, errors)
    }
  }

  override protected def parseManagementDependency(
      descriptor: DescriptorParseResult,
      dependency: ProjectDependency,
      dependenciesInPath: Set[Gav],
      loader: ResourceLoader,
      parentClassifier: Option[String]
  ): Future[Dependency] = {
    // We don't care about management dependencies for legacy descriptors
    val DescriptorParseResult(project, _) = descriptor
    Future.successful(IgnoredDependency(project, Nil))
  }

  private def scopeErrors(
      descriptor: ProjectDescriptor,
      dependency: ProjectDependency,
      results: Seq[APBResult],
      isRoot: Boolean
  ): Seq[APBResult] = {
    if (dependency.hasExplicitScope && dependency.scope != DesignScope) {
      results :+ scopeError(descriptor, dependency, isRoot = isRoot)
    } else results
  }

  private def scopeError(descriptor: ProjectDescriptor, dependency: ProjectDependency, isRoot: Boolean) = {
    val location = if (isRoot) DescriptorErrorLocation.forRoot() else DescriptorErrorLocation(dependency)
    new APBResult(
        s"Dependency ${dependency.gav.path} in module ${descriptor.gav.path} shouldn't have a defined scope in a 0.1.0 project",
        SeverityLevels.WARNING,
        Right(dependency.gav.path()),
        None,
        ScopePresentInLegacyProject.id,
        None,
        Some(location),
        Unit,
        ProjectAspects.TREE
    )
  }
}
