package org.mulesoft.apb.project.internal.emitter

import amf.core.client.scala.model.document.BaseUnit
import amf.core.client.scala.model.domain.{AmfElement, AmfObject, AmfScalar}
import amf.core.internal.metamodel.Field
import amf.core.internal.parser.domain.FieldEntry
import amf.core.internal.render.BaseEmitters.ScalarEmitter
import amf.core.internal.render.emitters.PartEmitter
import amf.shapes.client.scala.model.domain.jsonldinstance.JsonLDObject
import org.yaml.model.YDocument.EntryBuilder
import org.yaml.model.YType.Str
import org.yaml.model.{YNode, YType}

object EmitterUtils {
  private def emitField(
      entryBuilder: EntryBuilder,
      `object`: AmfObject,
      field: Field,
      nameOverride: Option[String] = None
  )(
      createPartEmitterFn: FieldEntry => PartEmitter
  ): Unit =
    for {
      fieldEntry <- `object`.fields.entry(field)
      valueEmitter = createPartEmitterFn(fieldEntry)
      actualName   = nameOverride.getOrElse(field.doc.displayName)
    } yield {
      entryBuilder.entry(YNode(actualName), valueEmitter.emit _)
    }

  def emitScalarField(
      entryBuilder: EntryBuilder,
      `object`: AmfObject,
      field: Field,
      nameOverride: Option[String] = None,
      `type`: YType = Str
  ): Unit = emitField(entryBuilder, `object`, field, nameOverride) { entry =>
    ScalarEmitter(entry.scalar, `type`)
  }

  def emitObjectField(
      entryBuilder: EntryBuilder,
      `object`: AmfObject,
      field: Field,
      nameOverride: Option[String] = None
  )(
      createPartEmitterFn: AmfObject => PartEmitter
  ): Unit = emitField(entryBuilder, `object`, field, nameOverride) { entry =>
    createPartEmitterFn(entry.obj)
  }

  def emitArrayField(
      entryBuilder: EntryBuilder,
      `object`: AmfObject,
      field: Field,
      nameOverride: Option[String] = None
  )(
      createPartEmitterFn: PartialFunction[AmfElement, PartEmitter]
  ): Unit = for {
    fieldEntry <- `object`.fields.entry(field)
    itemEmitters =
      fieldEntry.array.values
        .collect(createPartEmitterFn)
    actualName = nameOverride.getOrElse(field.doc.displayName)
  } yield {
    entryBuilder.entry(
        YNode(actualName),
        partBuilder => {
          partBuilder.list { arrayBuilder =>
            itemEmitters.foreach { itemEmitter =>
              itemEmitter.emit(arrayBuilder)
            }
          }
        }
    )
  }

  def emitObjectArrayField(
      entryBuilder: EntryBuilder,
      `object`: AmfObject,
      field: Field,
      nameOverride: Option[String] = None
  )(
      createPartEmitterFn: AmfObject => PartEmitter
  ): Unit = emitArrayField(entryBuilder, `object`, field, nameOverride) { case arrayObject: AmfObject =>
    createPartEmitterFn(arrayObject)
  }

  def emitScalarArrayField(
      entryBuilder: EntryBuilder,
      `object`: AmfObject,
      field: Field,
      nameOverride: Option[String] = None,
      `type`: YType = Str
  ): Unit = emitArrayField(entryBuilder, `object`, field, nameOverride) { case arrayScalar: AmfScalar =>
    ScalarEmitter(arrayScalar, `type`)
  }

  def emitJsonLdObject(
      entryBuilder: EntryBuilder,
      `object`: AmfObject,
      field: Field,
      nameOverride: Option[String] = None
  )(
      createPartEmitterFn: JsonLDObject => PartEmitter
  ): Unit = emitObjectField(entryBuilder, `object`, field, nameOverride) { case jsonLDObject: JsonLDObject =>
    createPartEmitterFn(jsonLDObject)
  }

  def emitJsonLdObjectArray(
      entryBuilder: EntryBuilder,
      `object`: AmfObject,
      field: Field,
      nameOverride: Option[String] = None
  )(
      createPartEmitterFn: JsonLDObject => PartEmitter
  ): Unit = emitArrayField(entryBuilder, `object`, field, nameOverride) { case jsonLDObject: JsonLDObject =>
    createPartEmitterFn(jsonLDObject)
  }

  def emitBaseUnitField(
      entryBuilder: EntryBuilder,
      `object`: AmfObject,
      field: Field,
      nameOverride: Option[String] = None
  )(
      createPartEmitterFn: BaseUnit => PartEmitter
  ): Unit = emitField(entryBuilder, `object`, field, nameOverride) { fieldEntry =>
    fieldEntry.value.value match {
      case baseUnit: BaseUnit => createPartEmitterFn(baseUnit)
    }
  }

  // For values that are not part of the AMF model but are part of the YAML model (e.g. tuple)
  def emitStringValue(entryBuilder: EntryBuilder, fieldName: String, value: String): Unit =
    entryBuilder.entry(YNode(fieldName), value)
}
