package org.mulesoft.apb.project.client.scala.model.project.management

import amf.shapes.client.scala.model.domain.jsonldinstance.JsonLDObject
import amf.shapes.internal.domain.metamodel.jsonldschema.JsonLDEntityModel
import amf.shapes.internal.spec.jsonldschema.parser.JsonPath
import org.mulesoft.apb.project.client.scala.model.DynamicObject
import org.mulesoft.apb.project.client.scala.model.project.management.APIInstance._
import org.mulesoft.apb.project.client.scala.model.project.management.SchemaIris.{
  ALERTS,
  ENVIRONMENT,
  INSTANCE_ADDRESS,
  POLICIES,
  SPEC
}
import org.mulesoft.apb.project.internal.instances.APIInstanceModel.API_INSTANCE_TERM
import org.mulesoft.apb.project.internal.model.GraphAccessors

case class APIInstance(override private[apb] val internal: JsonLDObject)
    extends GraphAccessors
    with DynamicObject
    with HasName
    with HasMetadata
    with HasAnnotations
    with HasLabels
    with HasSpec {
  def withService(name: String, address: String): Service = withPolicyBinding().withService(name, address)

  def services: Seq[Service] = policyBindings.flatMap(pb => pb.serviceIfPresent)

  def address: String = {
    spec.getScalar[String](INSTANCE_ADDRESS)
  }

  def instanceId: Option[Int] = getLabelIfPresent[Int](INSTANCE_ID_LABEL_NAME)

  def instanceLabel: Option[String] = getLabelIfPresent[String](INSTANCE_LABEL_LABEL_NAME)

  def environmentId: Option[String] = getLabelIfPresent[String](ENVIRONMENT_ID_LABEL_NAME)

  def consumerEndpoint: Option[String] = getLabelIfPresent[String](CONSUMER_ENDPOINT_LABEL_NAME)

  def hasApprovedContracts: Option[Boolean] = getLabelIfPresent[Boolean](HAS_APPROVED_CONTRACTS_LABEL_NAME)

  def withInstanceId(id: Int) = addLabel(INSTANCE_ID_LABEL_NAME, id)

  def withInstanceLabel(label: String) = addLabel(INSTANCE_LABEL_LABEL_NAME, label)

  def withEnvironmentId(envId: String) = addLabel(ENVIRONMENT_ID_LABEL_NAME, envId)

  def withConsumerEndpoint(endpoint: String) = addLabel(CONSUMER_ENDPOINT_LABEL_NAME, endpoint)

  def withApprovedContracts(hasApprovedContracts: Boolean): this.type =
    addLabel(HAS_APPROVED_CONTRACTS_LABEL_NAME, hasApprovedContracts)

  def withAddress(address: String): this.type = {
    update(SPEC) { spec =>
      spec.withProperty(INSTANCE_ADDRESS, address)
    }
    this
  }

  def alerts(): Seq[Alert] = spec.getObjectArray(ALERTS).map(Alert(_))

  def withAlerts(alerts: Seq[Alert]) = update(SPEC) { spec =>
    spec.withProperty(ALERTS, alerts)
  }

  def policyBindings: Seq[PolicyBinding] = {
    spec.getObjectArray(POLICIES).map(obj => PolicyBinding(obj.internal))
  }

  def withPolicyBinding(): PolicyBinding = {
    val policyBinding = PolicyBinding(name)
    update(SPEC) { spec =>
      spec.withProperty(POLICIES, policyBindings :+ policyBinding)
    }
    policyBinding
  }

  def clearPolicyBindings(): this.type = {
    update(SPEC) { spec =>
      spec.removeProperty(POLICIES)
    }
    this
  }

  def withPolicyBindings(policies: Seq[PolicyBinding]): this.type = {
    update(SPEC) { spec =>
      spec.withProperty(POLICIES, policyBindings ++ policies)
    }
    this
  }

  def environment: Environment = Environment(spec.getObject(ENVIRONMENT))

  def environmentOption: Option[Environment] =
    specIfPresent.flatMap(_.getObjectIfPresent(ENVIRONMENT)).map(Environment(_))

  def withEnvironment(name: String): Environment = {
    val environment = Environment(name)
    update(SPEC) { spec =>
      spec.withProperty(ENVIRONMENT, environment)
    }
    environment
  }
}

object APIInstance {
  val INSTANCE_ID_LABEL_NAME            = "mulesoft.com/apiinstance-id"
  val INSTANCE_LABEL_LABEL_NAME         = "mulesoft.com/apiinstance-label"
  val ENVIRONMENT_ID_LABEL_NAME         = "mulesoft.com/environment-id"
  val CONSUMER_ENDPOINT_LABEL_NAME      = "mulesoft.com/apiinstance-consumer-endpoint"
  val HAS_APPROVED_CONTRACTS_LABEL_NAME = "mulesoft.com/apiinstance-has-approved-contracts"

  def apply(name: String, address: String): APIInstance = {
    val path  = JsonPath.empty
    val model = JsonLDEntityModel(List(API_INSTANCE_TERM), Nil, path)
    val init  = JsonLDObject.empty(model, path)
    APIInstance(init)
      .withProperty("http://a.ml/vocabularies/core#apiVersion", Constants.apiVersion)
      .withProperty("http://a.ml/vocabularies/core#kind", Constants.apiInstanceKind)
      .withName(name)
      .withAddress(address)
  }

  private[apb] def apply(internal: JsonLDObject) = new APIInstance(internal)
}
