package org.mulesoft.apb.project.client.scala.model.project.management

import amf.shapes.client.scala.model.domain.jsonldinstance.JsonLDObject
import amf.shapes.internal.domain.metamodel.jsonldschema.JsonLDEntityModel
import amf.shapes.internal.spec.jsonldschema.parser.JsonPath
import org.mulesoft.apb.project.client.scala.model.DynamicObject
import org.mulesoft.apb.project.client.scala.model.descriptor.Gav
import org.mulesoft.apb.project.client.scala.model.project.management.PolicyBinding.DEFAULT_RULE_VALUE
import org.mulesoft.apb.project.client.scala.model.project.management.SchemaIris.{
  ALERTS,
  CONFIG,
  DEFINITION,
  DESTINATION_REF,
  MANAGEMENT,
  NAME,
  ORDER,
  POLICY_BINDING_RULES,
  POLICY_REF,
  SERVICE,
  SPEC
}
import org.mulesoft.apb.project.internal.instances.APIInstanceModel.POLICY_BINDING_TERM
import org.mulesoft.apb.project.internal.model.GraphAccessors
import org.mulesoft.apb.project.internal.model.GraphAccessors.toJsonLDObject

import scala.reflect.ClassTag

/** PolicyBinding, either inlined or standalone resource. TODO: implement getters and setters for standalone mode
  * @param init
  *   initial value
  * @param inlined
  *   indicates if this is an inlined PolicyBindings. InlinedBindings do not have a 'SPEC' object, they are directly the
  *   'SPEC' object
  */
case class PolicyBinding(override private[apb] val internal: JsonLDObject)
    extends GraphAccessors
    with DynamicObject
    with HasTargetRef
    with HasAnnotations
    with HasLabels
    with HasName {

  def service: Service = {
    val service = spec.getObject(CONFIG).getObject(SERVICE)
    Service(service)
  }

  def serviceIfPresent: Option[Service] = {
    specIfPresent.flatMap(_.getObjectIfPresent(CONFIG)).flatMap(_.getObjectIfPresent(SERVICE)).map(Service(_))
  }

  def withService(name: String, address: String): Service = {
    val service = Service(name, address)
    withService(service)
    service
  }

  def withService(service: Service): this.type = {
    update(SPEC) { spec =>
      spec.update(CONFIG) { config =>
        config.withProperty(SERVICE, service)
      }
    }
    this
  }

  def alerts(): Seq[Alert] = specIfPresent.toSeq.flatMap(_.getObjectArray(ALERTS)).map(Alert(_))

  def withAlerts(alerts: Seq[Alert]): this.type = {
    update(SPEC) { spec =>
      spec.withProperty(ALERTS, alerts)
    }
    this
  }

  def order: Int = spec.getScalar[Int](ORDER)

  def withOrder(order: Int): this.type = {
    update(SPEC) { spec =>
      spec.withProperty(ORDER, order)
    }
    this
  }

  def withExtension(extension: Extension, gav: Gav): this.type = {
    update(SPEC) { spec =>
      extension.withGav(gav)
      spec.withProperty(DEFINITION, extension)
      // Need this to validate. I suppose is removed when extracted because thought it was needed anymore. Only JSON-LD changes
//      spec.internal.graph.removeField(POLICY_REF)
    }
    this
  }

  def extension: Extension = Extension(spec.getObject(DEFINITION).internal)

  def policyRef: PolicyRef = {
    val ref = spec.getObject(POLICY_REF)
    PolicyRef(ref.internal)
  }

  def withPolicyRef(name: String, namespace: Option[String] = None): this.type = {
    update(SPEC) { spec =>
      val init = PolicyRef(name)
      namespace.foreach(init.withNamespace)
      spec.withProperty(POLICY_REF, init)
    }
    this
  }

  private[apb] def getPolicyRefName: Option[String] =
    specIfPresent
      .flatMap(_.getObjectIfPresent(POLICY_REF))
      .flatMap(_.getScalarIfPresent[String](NAME))

  private[apb] def getDestinationRefName: Option[String] = spec
    .getObjectIfPresent(CONFIG)
    .flatMap(_.getObjectIfPresent(DESTINATION_REF))
    .flatMap(_.getScalarIfPresent[String](NAME))

  def getScalarConfig[T](name: String)(implicit ct: ClassTag[T]): T = {
    val config = spec.getObject(CONFIG)
    config.getScalar[T](s"$MANAGEMENT#$name")
  }

  def getObjectConfig(name: String): GraphAccessors = {
    val config = spec.getObject(CONFIG)
    config.getObject(s"$MANAGEMENT#$name")
  }

  def getConfig: JsonLDObject = toJsonLDObject(spec.getObject(CONFIG))

  def getConfigOption: Option[JsonLDObject] = spec.getObjectIfPresent(CONFIG).map(toJsonLDObject)

  private def updateConfig(fn: (GraphAccessors => GraphAccessors)): this.type = {
    update(SPEC) { spec =>
      spec.update(CONFIG) { config =>
        fn(config)
      }
    }
    this
  }

  def withConfig(name: String, value: String): this.type = {
    updateConfig((config: GraphAccessors) => config.withProperty(s"$MANAGEMENT#$name", value))
  }

  def withConfig(name: String, value: Int): this.type = {
    updateConfig((config: GraphAccessors) => config.withProperty(s"$MANAGEMENT#$name", value))
  }

  def withConfig(name: String, value: Boolean): this.type = {
    updateConfig((config: GraphAccessors) => config.withProperty(s"$MANAGEMENT#$name", value))
  }

  def withConfig(name: String, value: Float): this.type = {
    updateConfig((config: GraphAccessors) => config.withProperty(s"$MANAGEMENT#$name", value))
  }

  def withConfig(name: String, value: Double): this.type = {
    updateConfig((config: GraphAccessors) => config.withProperty(s"$MANAGEMENT#$name", value))
  }

  def withConfig(name: String, value: GraphAccessors with DynamicObject): this.type = {
    updateConfig((config: GraphAccessors) => config.withProperty(s"$MANAGEMENT#$name", value))
  }

  def withConfig(name: String, values: List[String]): this.type = {
    updateConfig((config: GraphAccessors) => config.withStringPropertyCollection(s"$MANAGEMENT#$name", values))
  }

  def rules: Seq[PolicyBindingRule] = {
    getObjectIfPresent(SPEC).map { spec =>
      val rules = spec.getObjectArray(POLICY_BINDING_RULES)
      rules.zipWithIndex
        .map { case (ruleDef, index) =>
          PolicyBindingRule(ruleDef)(index)
        }
    } getOrElse (Nil)

  }
  def withRule(
      path: String = DEFAULT_RULE_VALUE,
      method: String = DEFAULT_RULE_VALUE,
      host: String = DEFAULT_RULE_VALUE
  ): PolicyBindingRule = {
    val startingRules = rules
    val init          = PolicyBindingRule(path, method, host)(startingRules.length)
    update(SPEC) { spec =>
      spec.withProperty(POLICY_BINDING_RULES, startingRules :+ init)
    }
    init
  }
}

object PolicyBinding {
  val DEFAULT_RULE_VALUE                           = ".*"
  def apply(internal: JsonLDObject): PolicyBinding = new PolicyBinding(internal)

  def apply(name: String): PolicyBinding = {
    val path  = JsonPath.empty
    val model = JsonLDEntityModel(List(POLICY_BINDING_TERM), Nil, path)
    val init  = JsonLDObject.empty(model, path)
    PolicyBinding(init)
      .withProperty("http://a.ml/vocabularies/core#apiVersion", Constants.apiVersion)
      .withProperty("http://a.ml/vocabularies/core#kind", Constants.policyBindingKind)
      .withName(name)
  }
}
