package org.mulesoft.apb.project.internal.engine

import amf.apicontract.client.scala.{AMFConfiguration, APIConfiguration}
import amf.core.client.common.validation.SeverityLevels
import amf.core.client.scala.AMFParseResult
import amf.core.client.scala.model.document.{BaseUnit, Document, Module}
import amf.core.client.scala.resource.ResourceLoader
import amf.core.client.scala.validation.AMFValidationResult
import amf.core.internal.remote.Spec
import org.mulesoft.apb.project.client.scala.ProjectConfiguration
import org.mulesoft.apb.project.client.scala.config.AMFConfigurationFactory
import org.mulesoft.apb.project.client.scala.model.descriptor.ProjectDescriptor
import org.mulesoft.apb.project.internal.dependency.UnreachableDependency
import AMFConfigurationFactory.{createFromSpec, createSpecificationParsingConfig}
import org.mulesoft.apb.project.client.scala.dependency.UnitCacheBuilder
import org.mulesoft.apb.project.internal.parser.{APBEnv, SyncJsonLdSchemaParser}
import org.mulesoft.apb.project.internal.validations.ProjectValidations.{MissingMainFile, UnreacheableAsset}

import scala.concurrent.ExecutionContext.Implicits.global
import scala.concurrent.Future

class MainFileInfo(
    val loader: ResourceLoader,
    val descriptor: ProjectDescriptor,
    val config: ProjectConfiguration,
    val result: AMFParseResult,
    val parentClassifier: Option[String]
) {
  val baseUnit: BaseUnit = result.baseUnit

  def parseCompanionLibrary: Future[Option[(String, Module)]] = {
    parentClassifier.flatMap {
      case v if v.contains("raml") => Some((Spec.RAML10, ".raml"))
//      case Spec.GRAPHQL => ".graphql"
//      case Spec.GRPC => ".proto3"
      case _ => Some((Spec.RAML10, ".raml")) // todo: current semex classifier is aml? need previous classifier?
    } map {
      case (spec, extension) if descriptor.main.isDefined =>
        val companionName = descriptor.main.get.substring(0, descriptor.main.get.lastIndexOf(".")) + extension
        parseCompanion(spec, companionName).recoverWith { case _: Throwable => Future.successful(None) }
    } getOrElse (Future.successful(None))
  }

  private def parseCompanion(spec: Spec, companionName: String) = {
    APIConfiguration
      .fromSpec(spec) // todo: check for graphql and grpc later on
      .withResourceLoader(loader)
      .baseUnitClient()
      .parseLibrary(companionName)
      .map(l => Some((companionName, l.library)))
  }

}

object MainFileInfoBuilder {
  def build(
      loader: ResourceLoader,
      descriptor: ProjectDescriptor,
      config: ProjectConfiguration,
      parentClassifier: Option[String],
      cacheBuilder: UnitCacheBuilder
  ): Future[MainFileInfo] = {

    val amfConfig = createSpecificationParsingConfig(config, List(loader), cacheBuilder)
    // need to enforce listeners regarding project configuration value at AnypointTreeBuilder. Root descriptor version rules how to check paths.
    val withListeners = config.getAnypointConfigListeners.foldLeft(amfConfig) { (acc, curr) =>
      acc.withEventListener(curr)
    }
    descriptor.main match {
      case Some(mainFile) => parseWithMain(config, withListeners, mainFile, loader, descriptor, parentClassifier)
      case _              => Future.successful(errorMainFileInfo(loader, descriptor, config, parentClassifier))
    }
  }

  private def parseWithMain(
      config: ProjectConfiguration,
      amfConfig: AMFConfiguration,
      main: String,
      loader: ResourceLoader,
      descriptor: ProjectDescriptor,
      parentClassifier: Option[String]
  ) = {
    amfConfig
      .baseUnitClient()
      .parse(APBEnv.projectProtocol + main)
      .recoverWith({ case e: Throwable =>
        Future.successful(new AMFParseResult(UnreachableDependency.emptyUnit, Seq(notFoundResult(e, descriptor))))
      })
      .map(new MainFileInfo(loader, descriptor, config, _, parentClassifier))
  }

  def buildForManagement(
      loader: ResourceLoader,
      descriptor: ProjectDescriptor,
      config: ProjectConfiguration,
      parentClassifier: Option[String],
      cacheBuilder: UnitCacheBuilder
  ): Future[MainFileInfo] = {
    // In the case of a management dependency, main file is a manifest.yaml
    val amfConfig = createFromSpec(config, Spec.JSONSCHEMA, List(loader), cacheBuilder)
    parsePolicyDependency(config, amfConfig, loader, descriptor, parentClassifier)
  }

  private def parsePolicyDependency(
      config: ProjectConfiguration,
      amfConfig: AMFConfiguration,
      loader: ResourceLoader,
      descriptor: ProjectDescriptor,
      parentClassifier: Option[String]
  ): Future[MainFileInfo] = {
    parseWithMain(config, amfConfig, "policy-definition.json", loader, descriptor, parentClassifier)
  }

  private def notFoundResult(e: Throwable, descriptor: ProjectDescriptor): AMFValidationResult =
    AMFValidationResult(
      e.getMessage,
      SeverityLevels.VIOLATION,
      "",
      None,
      UnreacheableAsset.id,
      None,
      Some(descriptor.gav.path + APBEnv.descriptorFileName),
      Unit
    )

  private def errorMainFileInfo(
      loader: ResourceLoader,
      descriptor: ProjectDescriptor,
      config: ProjectConfiguration,
      parentClassifier: Option[String]
  ) = {
    val validation = AMFValidationResult(
      MissingMainFile.message,
      SeverityLevels.VIOLATION,
      "",
      None,
      MissingMainFile.id,
      None,
      Some(descriptor.gav.path + APBEnv.descriptorFileName),
      Unit
    )
    InvalidMainFileInfo(validation, loader, descriptor, config, parentClassifier)
  }
}

case class InvalidMainFileInfo(
    validation: AMFValidationResult,
    override val loader: ResourceLoader,
    override val descriptor: ProjectDescriptor,
    override val config: ProjectConfiguration,
    override val parentClassifier: Option[String]
) extends MainFileInfo(
      loader,
      descriptor,
      config: ProjectConfiguration,
      new AMFParseResult(Document().withId("amf://error"), Seq(validation)),
      parentClassifier
    )
