package org.mulesoft.apb.project.internal.transformation.steps

import amf.core.client.scala.AMFGraphConfiguration
import amf.core.client.scala.errorhandling.AMFErrorHandler
import amf.core.client.scala.model.document.BaseUnit
import amf.core.client.scala.model.domain.{AmfElement, AmfScalar}
import amf.core.client.scala.transform.TransformationStep
import amf.core.client.scala.traversal.iterator.AmfElementStrategy
import amf.core.client.scala.vocabulary.Namespace
import amf.shapes.client.scala.model.document.JsonLDInstanceDocument
import amf.shapes.client.scala.model.domain.jsonldinstance.JsonLDObject
import org.mulesoft.apb.project.client.scala.DependencySet
import org.mulesoft.apb.project.client.scala.model.descriptor.Metadata
import org.mulesoft.apb.project.internal.common.Classifier
import org.mulesoft.apb.project.internal.model.GraphAccessors.toJsonLDObject
import org.mulesoft.apb.project.internal.model.descriptor.MetadataModel
import org.mulesoft.apb.project.internal.validations.AgentNetworkValidations.{
  InvalidMetadataVariable,
  UnusedMetadataVariable
}

import scala.util.matching.Regex

class MetadataVariablesValidationStage(dependencySet: DependencySet) extends TransformationStep() {

  private val IGNORED_VARIABLES: Seq[String] = Seq("ingressgw.url", "egressgw.url")
  private val DESCRIPTION_FIELD: String      = s"${Namespace.Core.base}description"
  private val DEFAULT_FIELD: String          = s"${Namespace.Core.base}default"
  private val SECRET_FIELD: String           = s"${Namespace.Core.base}secret"

  override def transform(
      model: BaseUnit,
      errorHandler: AMFErrorHandler,
      configuration: AMFGraphConfiguration
  ): BaseUnit = {
    model match {
      case jsonDoc: JsonLDInstanceDocument if applies =>
        val metadataVariables: Seq[MetadataVariable] = parseVariables()
        val variableElements: Seq[VariableElement] =
          jsonDoc.iterator(strategy = AmfElementStrategy).toStream.flatMap(elementsWithVariables)

        validateVariables(metadataVariables, variableElements, errorHandler)
    }
    model
  }

  private def validateVariables(
      variables: Seq[MetadataVariable],
      elements: Seq[VariableElement],
      eh: AMFErrorHandler
  ): Unit = {
    // The validation has 2 steps
    // All variable present in the elements must be declared, if not is an error
    val variablesSet = variables.map(_.name) // This is to make faster the search
    elements.foreach { element =>
      element.variables.foreach { elementVariable =>
        if (!variablesSet.contains(elementVariable) && !isIgnoredVariable(elementVariable)) {
          eh.violation(
            InvalidMetadataVariable,
            "",
            s"The metadata variable '$elementVariable' is not declared",
            element.element.annotations
          )
        }
      }
    }
    // All variables declared in the descriptor should be used, if not is a warning
    val elementVariables: Set[String] =
      elements.flatMap(_.variables).toSet // all distinct variables used in the elements
    variables.foreach { variable =>
      if (!elementVariables.contains(variable.name)) {
        eh.warning(
          UnusedMetadataVariable,
          variable.node,
          s"The metadata variable '${variable.name}' is never used",
          variable.node.annotations.sourceLocation
        )
      }
    }
  }

  private def isIgnoredVariable(variable: String): Boolean = IGNORED_VARIABLES.contains(variable)

  private def applies: Boolean = dependencySet.descriptor().classifier().contains(Classifier.AGENT_NETWORK)

  private def parseVariables(): Seq[MetadataVariable] = {
    val metadata: Option[Metadata] = dependencySet.descriptor().metadata
    val metadataVariables: Option[JsonLDObject] =
      metadata.flatMap(_.getObjectIfPresent(MetadataModel.Variables.value.iri())).map(toJsonLDObject)
    metadataVariables.map(variables => traverseVariableNodes(variables, "")).toSeq.flatten
  }

  private def traverseVariableNodes(obj: JsonLDObject, parentPath: String): Seq[MetadataVariable] = {
    // The node is a variable or is a nested node which fields needs to be traversed
    if (isVariable(obj)) {
      Seq(parseVariable(obj, parentPath))
    } else {
      obj.fields
        .fields()
        .flatMap { variable =>
          val key = variable.field.value.name
          variable.value.value match {
            case obj: JsonLDObject => traverseVariableNodes(obj, concatPath(parentPath, key))
            case _                 => Nil // scalar case should be handled by the IF and array should not be accepted
          }
        }
        .toSeq
    }
  }

  private def parseVariable(obj: JsonLDObject, name: String): MetadataVariable = {

    val description = obj.fields.getValueAsOption(DESCRIPTION_FIELD).map(_.toString)
    val default     = obj.fields.getValueAsOption(DEFAULT_FIELD).map(_.toString)
    val secret =
      obj.fields.getValueAsOption(SECRET_FIELD).map(_.value).collect { case scalar: AmfScalar => scalar.toBool }
    MetadataVariable(name, description, default, secret, obj)
  }

  // Returns a boolean indicating if the node is a variable node and not a nested one
  private def isVariable(obj: JsonLDObject): Boolean =
    obj.fields.fields().forall(_.value.value.isInstanceOf[AmfScalar])

  private def concatPath(base: String, value: String): String = if (base.isEmpty) value else s"$base.$value"

  private def elementsWithVariables(amfElement: AmfElement): Option[VariableElement] = {
    amfElement match {
      case scalar: AmfScalar if hasVariables(scalar) =>
        val variables = getContentVariables(scalar.toString)
        Some(VariableElement(scalar, variables))
      case _ => None // Variables could be only used in scalar values
    }
  }

  private def hasVariables(scalar: AmfScalar): Boolean = scalar.toString.contains("${")

  private def getContentVariables(content: String): Seq[String] = {
    val pattern: Regex = """\$\{([^}]+)\}""".r
    pattern.findAllMatchIn(content).map(_.group(1)).toSeq
  }

  private case class MetadataVariable(
      name: String,
      description: Option[String],
      default: Option[String],
      secret: Option[Boolean],
      node: JsonLDObject
  )

  private case class VariableElement(element: AmfScalar, variables: Seq[String])
}
