package org.mulesoft.apb.client.scala

import amf.apicontract.client.scala.{AMFConfiguration, WebAPIConfiguration}
import amf.core.client.common.transform.PipelineId
import amf.core.client.scala.config.RenderOptions
import amf.core.client.scala.model.document.BaseUnit
import amf.core.client.scala.resource.ResourceLoader
import amf.core.client.scala.validation.{AMFValidationReport, AMFValidationResult}
import amf.core.client.scala.{AMFGraphConfiguration, AMFParseResult, AMFResult}
import amf.core.internal.remote.Spec
import amf.core.internal.unsafe.PlatformSecrets
//import amf.validation.client.ProfileValidatorNodeBuilder
//import amf.validation.internal.report.parser.EmptyProfile
import org.mulesoft.apb.client.scala.dependency.{
  AMFConfigBuilder,
  APBUnitCacheBuilder,
  ProjectConfiguration,
  UnitCacheBuilder
}
import org.mulesoft.apb.client.scala.environment.DependencyFetcher
import org.mulesoft.apb.client.scala.model.{Gav, ProjectDescriptor}
import org.mulesoft.apb.internal.parser.APBEnv

import scala.concurrent.ExecutionContext.Implicits.global
import scala.concurrent.Future
import scala.language.postfixOps

class APBClient(val descriptor: ProjectDescriptor,
                dependencyFetcher: DependencyFetcher,
                unitCacheBuilder: UnitCacheBuilder = APBUnitCacheBuilder,
                resourceLoaders: List[ResourceLoader]) {

  val projectConfiguration: Future[ProjectConfiguration] =
    new AMFConfigBuilder(dependencyFetcher, unitCacheBuilder, resourceLoaders).build(descriptor)

  implicit class SpecHelper(spec: Spec) {
    val config: AMFConfiguration = WebAPIConfiguration.fromSpec(spec)
  }

  implicit class AMFHelper(config: AMFConfiguration) {
    def parse(): Future[AMFParseResult]  = config.baseUnitClient().parse(descriptor.main)
    def resolve(bu: BaseUnit): AMFResult = config.baseUnitClient().transform(bu, PipelineId.Editing)

    def validate(bu: BaseUnit): Future[AMFValidationReport] = config.baseUnitClient().validate(bu)
  }

  implicit class AMFResultHelper(result: AMFResult) {
    def addResults(other: AMFResult): AMFResult = result.copy(results = result.results ++ other.results)
  }

  private def parse(): Future[(AMFParseResult, Spec)] = {
    for {
      config <- projectConfiguration.map(_.parseConfig)
      u      <- config.parse()
    } yield (u, u.sourceSpec)
  }

  private def resolve(r: AMFResult, spec: Spec) = projectConfiguration.map(_.configFor(spec).resolve(r.baseUnit))

  private def validate(r: AMFResult, spec: Spec) =
    projectConfiguration.flatMap(_.configFor(spec).validate(r.baseUnit.cloneUnit()))

  private def serialize(bu: BaseUnit, withSourceMaps: Boolean = false): Future[String] = {
    val options = if (withSourceMaps) RenderOptions().withSourceMaps else RenderOptions()
    projectConfiguration.map(
      _.dependenciesConfig(WebAPIConfiguration.WebAPI().withRenderOptions(options)).baseUnitClient().render(bu))
  }

//  private def lint(unit: BaseUnit): Future[AMFValidationReport] =
//    for {
//      jsonld <- serialize(unit, withSourceMaps = true)
//      reports <- projectConfiguration.flatMap { pc =>
//        val eventualReports: Seq[Future[AMFValidationReport]] = pc.profileDependencies.toList
//          .map { profileDep =>
//            ProfileValidatorNodeBuilder
//              .validator(profileDep.profile)
//              .validate(jsonld, descriptor.main)
//          }
//        Future.sequence(eventualReports)
//      }
//    } yield {
//      val empty = AMFValidationReport.empty(unit.id, EmptyProfile)
//      reports.foldLeft(empty)((acc, curr) => acc.merge(curr))
//    }

  def getAMFConfiguration: Future[AMFConfiguration] = projectConfiguration.map(_.parseConfig)

  def build(): Future[AMFResult] = {
    for {
      (result, spec) <- parse()
    } yield {
      val transformResult = WebAPIConfiguration.fromSpec(spec).resolve(result.baseUnit)
      transformResult.copy(results = transformResult.results ++ result.results)
    }
  }

  def compile(): Future[AMFParseResult] = parse().map(_._1)

  def report(): Future[AMFValidationReport] = {
    for {
      p <- parse()
      r <- resolve(p._1, p._2)
      v <- {
        validate(r, p._2)
      }
//      l <- lint(r.baseUnit)
    } yield v.copy(results = v.results ++ p._1.results ++ r.results) //++ l.results)
  }

//  def lint(): Future[AMFValidationReport] = build().flatMap(b => lint(b.baseUnit))

  def serialize(): Future[String] =
    for {
      r <- build()
      s <- serialize(r.baseUnit)
    } yield s

  def checkTree(): Future[Seq[AMFValidationResult]] = projectConfiguration.map(_.errors.allErrors)

  lazy val listDependencies: Future[Seq[Gav]] = projectConfiguration.map(_.allDependencies.map(_.descriptor.gav))

}

object APBClient extends PlatformSecrets {
  def fromContent(content: String, dependencyFetcher: DependencyFetcher, base: Option[String]): APBClient = {
    APBClientBuilder(dependencyFetcher).build(content, base)
  }

  def fromDirectory(directory: String, dependencyFetcher: DependencyFetcher): Future[APBClient] = {
    val eventualContent = fetchDescriptorFile(directory)
    eventualContent.map(c => fromContent(c.stream.toString, dependencyFetcher, Some(directory)))
  }

  private def fetchDescriptorFile(directory: String) = {
    platform.fetchContent(directory + "/" + APBEnv.descriptorFileName, AMFGraphConfiguration.predefined())
  }
}
