package org.mulesoft.apb.client.scala.dependency

import amf.aml.client.scala.model.document.{Dialect, DialectInstance}
import amf.core.client.common.validation.SeverityLevels
import amf.core.client.scala.model.document.{BaseUnit, ExternalFragment}
import amf.core.client.scala.validation.AMFValidationResult
import amf.core.client.scala.vocabulary.Namespace
import org.mulesoft.apb.client.scala.dependency.ProjectValidations.UnreacheableAsset
import org.mulesoft.apb.client.scala.dependency.UnreachableDependency.emptyUnit
import org.mulesoft.apb.client.scala.model._

abstract class ParsedDependency(val baseUnit: BaseUnit, val descriptor: ProjectDescriptor, val errors: ProjectErrors) {
  val scope: DependencyScope
  val location: String = baseUnit.location().getOrElse(baseUnit.id)
}

case class DesignDependency(override val baseUnit: BaseUnit,
                            override val descriptor: ProjectDescriptor,
                            override val errors: ProjectErrors)
    extends ParsedDependency(baseUnit, descriptor, errors) {
  override val scope: DependencyScope = DesignScope
}

case class ExtensionDependency(dialect: Dialect,
                               override val descriptor: ProjectDescriptor,
                               override val errors: ProjectErrors)
    extends ParsedDependency(dialect, descriptor, errors) {
  override val scope: DependencyScope = ExtensionScope
}

case class ProfileDependency(profile: DialectInstance,
                             override val descriptor: ProjectDescriptor,
                             override val errors: ProjectErrors)
    extends ParsedDependency(profile, descriptor, errors) {
  override val scope: DependencyScope = ValidationScope

  val profileName: String = Option(profile.encodes)
    .flatMap(
      _.graph
        .scalarByProperty((Namespace("http://schema.org/") + "name").iri())
        .headOption
        .map(_.toString))
    .getOrElse("Unknown")
}

private[apb] object UnreachableDependency {
  protected val emptyUnit                               = ExternalFragment().withId("unreachable-unit")
  protected def descriptor(gav: Gav): ProjectDescriptor = ProjectDescriptor("", "", gav, None, Nil, Nil)

  def apply(dependency: ProjectDependency): ParsedDependency = {
    val validation = AMFValidationResult(s"Asset ${dependency.gav.path} is unreachable",
                                         SeverityLevels.VIOLATION,
                                         "",
                                         None,
                                         UnreacheableAsset.id,
                                         None,
                                         Some(dependency.gav.path),
                                         Unit)
    new UnreachableDependency(descriptor(dependency.gav),
                              ProjectErrors(projectErrors = List(validation)),
                              dependency.scope)
  }
}

private[apb] case class UnreachableDependency(override val descriptor: ProjectDescriptor,
                                              override val errors: ProjectErrors,
                                              override val scope: DependencyScope)
    extends ParsedDependency(emptyUnit, descriptor, errors) {}
