package org.mulesoft.apb.client.scala.dependency

import amf.aml.client.scala.model.document.{Dialect, DialectInstance}
import amf.apicontract.client.scala.{AMFConfiguration, WebAPIConfiguration}
import amf.core.client.scala.config.UnitCache
import amf.core.client.scala.resource.ResourceLoader
import amf.core.client.scala.validation.AMFValidationResult
import amf.core.internal.remote.Spec
import org.mulesoft.apb.client.scala.model.{Gav, ProjectDescriptor}

case class ProjectErrors(treeErrors: List[AMFValidationResult] = Nil, projectErrors: List[AMFValidationResult] = Nil) {
  private[apb] def addTreeErrors(errors: List[AMFValidationResult]) = copy(treeErrors = treeErrors ++ errors)
  private[apb] def addProjectError(error: AMFValidationResult)      = copy(projectErrors = error :: projectErrors)
  private[apb] def add(errors: ProjectErrors): ProjectErrors =
    copy(treeErrors = this.treeErrors ++ errors.treeErrors, projectErrors = this.projectErrors ++ errors.projectErrors)
  private[apb] def allErrors: List[AMFValidationResult] = treeErrors ::: projectErrors
}

class ProjectConfiguration(val designDependencies: Seq[DesignDependency],
                           val profileDependencies: Seq[ProfileDependency],
                           val extensionDependencies: Seq[ExtensionDependency],
                           val dependenciesLoaders: Map[String, ResourceLoader],
                           descriptor: ProjectDescriptor,
                           validationDialect: Dialect,
                           unitCacheBuilder: UnitCacheBuilder = APBUnitCacheBuilder,
                           val envLoaders: List[ResourceLoader],
                           val errors: ProjectErrors) {
  val mainFile: String = descriptor.main

  private val mainLoader: ResourceLoader = new APBResourceLoader(dependenciesLoaders)
  private lazy val unitsCache: UnitCache = unitCacheBuilder.build(designDependencies)

  lazy val parseConfig: AMFConfiguration = dependenciesConfig(WebAPIConfiguration.WebAPI())

  val allDependencies: Seq[ParsedDependency] = designDependencies ++ extensionDependencies ++ profileDependencies

  def dependenciesConfig(base: AMFConfiguration): AMFConfiguration = {
    val inc =
      base.withResourceLoaders(List(mainLoader) ++ envLoaders).withDialect(validationDialect).withUnitCache(unitsCache)
    extensionDependencies
      .foldLeft(inc) { (acc, curr) =>
        acc.withDialect(curr.dialect)
      }
  }

  def configFor(spec: Spec): AMFConfiguration = dependenciesConfig(WebAPIConfiguration.fromSpec(spec))

  def getDialectFor(instance: DialectInstance): Option[Dialect] = {
    parseConfig.configurationState().findDialectFor(instance)
  }
}
