package org.mulesoft.apb.client.scala

import amf.apicontract.client.scala.{AMFConfiguration, WebAPIConfiguration}
import amf.core.client.common.transform.PipelineId
import amf.core.client.scala.config.RenderOptions
import amf.core.client.scala.model.document.BaseUnit
import amf.core.client.scala.resource.ResourceLoader
import amf.core.client.scala.validation.{AMFValidationReport, AMFValidationResult}
import amf.core.client.scala.{AMFGraphConfiguration, AMFParseResult, AMFResult}
import amf.core.internal.remote.Spec
import amf.core.internal.unsafe.PlatformSecrets
import amf.custom.validation.client.ProfileValidatorNodeBuilder
import amf.custom.validation.internal.report.parser.EmptyProfile
import org.mulesoft.apb.project.client.scala.{ProjectBuilder, ProjectConfiguration}
import org.mulesoft.apb.project.client.scala.dependency.{APBUnitCacheBuilder, UnitCacheBuilder}
import org.mulesoft.apb.project.client.scala.environment.DependencyFetcher
import org.mulesoft.apb.project.client.scala.model.{Gav, ProjectDescriptor}
import org.mulesoft.apb.project.internal.parser.APBEnv
//import amf.validation.client.ProfileValidatorNodeBuilder
//import amf.validation.internal.report.parser.EmptyProfile
import scala.concurrent.ExecutionContext.Implicits.global
import scala.concurrent.Future
import scala.language.postfixOps

class APBClient(val projectConfiguration: ProjectConfiguration) {

  val descriptor: ProjectDescriptor = projectConfiguration.descriptor

  implicit class SpecHelper(spec: Spec) {
    val config: AMFConfiguration = WebAPIConfiguration.fromSpec(spec)
  }

  implicit class AMFHelper(config: AMFConfiguration) {
    def parse(): Future[AMFParseResult]  = config.baseUnitClient().parse(descriptor.main)
    def resolve(bu: BaseUnit): AMFResult = config.baseUnitClient().transform(bu, PipelineId.Editing)

    def validate(bu: BaseUnit): Future[AMFValidationReport] = config.baseUnitClient().validate(bu)
  }

  implicit class AMFResultHelper(result: AMFResult) {
    def addResults(other: AMFResult): AMFResult = result.copy(results = result.results ++ other.results)
  }

  private def parse(): Future[(AMFParseResult, Spec)] = {
    projectConfiguration.projectParseConfig.parse().map(u => (u, u.sourceSpec))
  }

  private def resolve(r: AMFResult, spec: Spec) = projectConfiguration.configFor(spec).resolve(r.baseUnit)

  private def validate(r: AMFResult, spec: Spec) =
    projectConfiguration.configFor(spec).validate(r.baseUnit.cloneUnit())

  private def serialize(bu: BaseUnit, withSourceMaps: Boolean = false): String = {
    val options = if (withSourceMaps) RenderOptions().withSourceMaps else RenderOptions()
    projectConfiguration
      .dependenciesConfig(WebAPIConfiguration.WebAPI().withRenderOptions(options))
      .baseUnitClient()
      .render(bu)
  }

  private def lint(unit: BaseUnit): Future[AMFValidationReport] =
    for {
      reports <- {
        val jsonld = serialize(unit, withSourceMaps = true)
        val eventualReports: Seq[Future[AMFValidationReport]] = projectConfiguration.profileDependencies.toList
          .map { profileDep =>
            ProfileValidatorNodeBuilder
              .validator(profileDep.profile)
              .validate(jsonld, descriptor.main)
          }
        Future.sequence(eventualReports)
      }
    } yield {
      val empty = AMFValidationReport.empty(unit.id, EmptyProfile)
      reports.foldLeft(empty)((acc, curr) => acc.merge(curr))
    }

  def getAMFConfiguration: AMFConfiguration = projectConfiguration.projectParseConfig

  def build(): Future[AMFResult] = {
    for {
      (result, spec) <- parse()
    } yield {
      val transformResult = WebAPIConfiguration.fromSpec(spec).resolve(result.baseUnit)
      transformResult.copy(results = transformResult.results ++ result.results)
    }
  }

  def compile(): Future[AMFParseResult] = parse().map(_._1)

  def report(): Future[AMFValidationReport] = {
    for {
      (p, r) <- parse().map(r => (r, resolve(r._1, r._2)))
      v <- {
        validate(r, p._2)
      }
      l <- lint(r.baseUnit)
    } yield v.copy(results = v.results ++ p._1.results ++ r.results ++ l.results)
  }

  def lint(): Future[AMFValidationReport] = build().flatMap(b => lint(b.baseUnit))

  def serialize(): Future[String] = build().map(r => serialize(r.baseUnit))

  def checkTree(): Seq[AMFValidationResult] = projectConfiguration.errors.allErrors

  lazy val listDependencies: Seq[Gav] = projectConfiguration.allDependencies.map(_.descriptor.gav)

}

object APBClient extends PlatformSecrets {
  def fromContent(content: String, dependencyFetcher: DependencyFetcher, base: Option[String]): Future[APBClient] = {
    APBClientBuilder(dependencyFetcher).build(content, base)
  }

  def fromDirectory(directory: String, dependencyFetcher: DependencyFetcher): Future[APBClient] = {
    APBClientBuilder(dependencyFetcher).build(directory)
  }
}
