package org.mulesoft.apb.client.scala

import amf.apicontract.client.scala.APIConfiguration
import amf.apicontract.client.scala.model.domain.api.Api
import amf.core.client.scala.AMFResult
import amf.core.client.scala.config.RenderOptions
import amf.core.client.scala.model.document.Document
import amf.core.client.scala.model.domain.DomainElement
import amf.core.client.scala.resource.ResourceLoader
import amf.core.internal.unsafe.PlatformSecrets
import amf.shapes.client.scala.config.JsonLDSchemaConfiguration
import amf.shapes.client.scala.model.domain.jsonldinstance.JsonLDObject
import org.mulesoft.apb.client.scala.model.{Project, ProjectBuilder}
import org.mulesoft.apb.internal.convert.ElementConverters.{AMFResultConverter, DomainElementConverter}
import org.mulesoft.apb.project.client.scala.environment.DependencyFetcher
import org.mulesoft.apb.project.client.scala.model.{Instance, ProjectDescriptor}
import org.mulesoft.apb.project.internal.descriptor.ApiProjectNamespaces
import org.mulesoft.apb.project.internal.descriptor.ApiProjectNamespaces.aliases

import scala.concurrent.ExecutionContext.Implicits.global
import scala.concurrent.Future
// TODO native-jsonld: change policies for a list of generic runtime dependencies with their own definition for merging
class APIProjectClient private[apb] (
    projectDescriptor: ProjectDescriptor,
    dependencyFetcher: DependencyFetcher,
    rl: Option[ResourceLoader],
    cache: Option[ProjectNodeCache]
) extends PlatformSecrets {

  def build(): Future[Project] = {
    val builder = ProjectBuilder(projectDescriptor)
    for {
      _ <- buildContract().map(builder.withContract)
      _ <- buildInstances().map(i => builder.withInstances(i.toList))
    } yield builder.build()
  }

  def serialize(): Future[String] = {
    build().map(p => buildConfig.baseUnitClient().render(p.document))
  }

  private def getCachedOrBuild(orElseFN: () => Future[DomainElement])(uri: String): Future[DomainElement] =
    cache.flatMap(_.fetch(uri)).getOrElse(orElseFN())

  private def buildContractFromDescriptor(): Future[DomainElement] = {
    val builder = ApiContractClientBuilder(dependencyFetcher)
    rl.foreach(builder.withResourceLoader)
    builder
      .build(projectDescriptor)
      .flatMap(_.build())
      .map(_.encodedElement())
  }

  private def buildInstance(instance: Instance): Future[JsonLDObject] = {
    val client = APIInstanceClient(instance.gcl, instance.definedBy, rl)
    getCachedOrBuild(client.build)(instance.gcl).map(_.toJsonLDObject())
  }

  private def buildInstances(): Future[Seq[JsonLDObject]] =
    Future.sequence(projectDescriptor.instances.map(buildInstance))

  private def buildContract(): Future[Api] =
    getCachedOrBuild(buildContractFromDescriptor)(projectDescriptor.main).map(_.toApi())

  private def buildConfig = {
    APIConfiguration
      .API()
      .withRenderOptions(RenderOptions().withPrettyPrint.withCompactUris)
      .withAliases(aliases)
  }
}

trait ProjectNodeCache {
  def fetch(uri: String): Option[Future[DomainElement]]
}
