package org.mulesoft.apb.client.scala.model

import amf.apicontract.client.scala.model.domain.api.Api
import amf.core.client.scala.model.document.{BaseUnit, EncodesModel}
import amf.core.internal.adoption.IdAdopter
import amf.core.internal.metamodel.document.BaseUnitModel
import amf.core.internal.metamodel.document.DocumentModel.Encodes
import amf.core.internal.parser.domain.{Annotations, Fields}
import amf.shapes.client.scala.model.domain.jsonldinstance.{JsonLDArray, JsonLDObject}
import org.mulesoft.apb.internal.metamodel.ProjectDocumentModel
import org.mulesoft.apb.internal.metamodel.ProjectInfoModel.{ApiContract, Instances}
import org.mulesoft.apb.project.client.scala.model.ProjectDependencyModel.{AssetId, GroupId, Version}
import org.mulesoft.apb.project.client.scala.model.ProjectDescriptor
import org.mulesoft.apb.project.client.scala.model.ProjectDescriptorModel.{Dependencies, OrganizationId}

class ProjectDocument(override val fields: Fields, override val annotations: Annotations)
    extends BaseUnit
    with EncodesModel {

  /** Meta data for the document */
  override def meta: BaseUnitModel = ProjectDocumentModel

  /** Returns the list document URIs referenced from the document that has been parsed to generate this model */
  override def references: Seq[BaseUnit] = Nil

  /** Encoded DomainElement described in the document element. */
  override def encodes: ProjectInfo = fields(Encodes)

  /** Value , path + field value that is used to compose the id when the object its adopted */
  override def componentId: String = "project"
}

object ProjectDocument {
  def apply() = new ProjectDocument(Fields(), Annotations())
}

case class Project(private[apb] val document: ProjectDocument) {

  private def projectInfo            = document.encodes
  def apiContract(): Api             = projectInfo.apiContract()
  def instances(): Seq[JsonLDObject] = projectInfo.instances()

  // TODO native-jsonld : add properties of the descriptor
}

case class ProjectBuilder(descriptor: ProjectDescriptor) {
  private val projectInfo     = ProjectInfo()
  private val projectDocument = ProjectDocument().set(Encodes, projectInfo)

  def withContract(api: Api): ProjectBuilder = {
    projectInfo.set(ApiContract, api)
    this
  }

  def withInstances(instances: List[JsonLDObject]): ProjectBuilder = {
    val arr = new JsonLDArray()
    instances.foreach(arr += _)
    projectInfo.set(Instances, arr)
    this
  }

  def build(): Project = {
    populateProjectWith(projectInfo, descriptor)
    new IdAdopter(projectDocument, "http://project.aml#").adoptFromRoot()
    Project(projectDocument)
  }

  private def populateProjectWith(projectInfo: ProjectInfo, descriptor: ProjectDescriptor) = {
    whenNonEmpty(descriptor.gav.groupId, groupId => projectInfo.set(GroupId, groupId))
    whenNonEmpty(descriptor.gav.assetId, assetId => projectInfo.set(AssetId, assetId))
    whenNonEmpty(descriptor.gav.version, version => projectInfo.set(Version, version))
    descriptor.organizationId.foreach(orgId => projectInfo.set(OrganizationId, orgId))
    projectInfo.setArray(Dependencies, descriptor.dependencies.map(_.wrapped))
  }
  private def whenNonEmpty[T](value: String, func: String => T): Option[T] = {
    if (value.nonEmpty) Some(func(value))
    else None
  }
}
