package org.mulesoft.apb.client.scala

import amf.apicontract.client.scala.APIConfiguration
import amf.core.client.scala.config.RenderOptions
import amf.core.client.scala.model.document.BaseUnit
import amf.core.client.scala.resource.ResourceLoader
import amf.core.internal.unsafe.PlatformSecrets
import org.mulesoft.apb.internal.convert.ElementConverters._
import org.mulesoft.apb.project.client.scala.environment.DependencyFetcher
import org.mulesoft.apb.project.client.scala.extensions.APIProjectExtension
import org.mulesoft.apb.project.client.scala.model.{Instance, Project, ProjectDocumentBuilder, ProjectDescriptor}
import org.mulesoft.apb.project.internal.BuildResult
import org.mulesoft.apb.project.internal.descriptor.ApiProjectNamespaces.aliases
import org.mulesoft.apb.project.internal.view.ApiSummaryView

import scala.concurrent.ExecutionContext.Implicits.global
import scala.concurrent.Future
// TODO native-jsonld: change policies for a list of generic runtime dependencies with their own definition for merging
class APIProjectClient private[apb] (
    projectDescriptor: ProjectDescriptor,
    dependencyFetcher: DependencyFetcher,
    rl: Option[ResourceLoader],
    cache: Option[ProjectNodeCache],
    extensions: Seq[APIProjectExtension],
    base: Option[String] = None
) extends PlatformSecrets {

  def build(): Future[BuildResult[Project]] = {
    val builder = ProjectDocumentBuilder(projectDescriptor).withExtensions(extensions)
    for {
      builtContract  <- buildContract()
      builtInstances <- buildInstances()
    } yield {
      val (instances, errors) = extractInstances(builtInstances)
      builder.withContract(builtContract.result, builtContract.report.results)
      builder.withInstances(instances, errors)
      base.foreach(builder.withBase)
      builder.build()
    }
  }

  private def extractInstances(units: Seq[BuildResult[BaseUnit]]) =
    (units.flatMap(_.result.toJsonLDObjects()).toList, units.flatMap(_.report.results))

  def serialize(): Future[String] = {
    build().map(p => buildConfig.baseUnitClient().render(p.result.document))
  }

  def serialize(project: Project): String = buildConfig.baseUnitClient().render(project.document)

  def summary(schemaBase: String): Future[String] = {
    val summary = ApiSummaryView(schemaBase)
    build().map(project => summary.view(project.result.projectInfo))
  }

  def summary(schemaBase: String, project: Project): String = {
    val summary = ApiSummaryView(schemaBase)
    summary.view(project.projectInfo)
  }

  private def getCachedOrBuild(
      orElseFN: () => Future[BuildResult[BaseUnit]]
  )(uri: String): Future[BuildResult[BaseUnit]] =
    cache
      .flatMap(_.fetch(uri))
      .map(future => future.map(unit => BuildResult(unit)))
      .getOrElse(orElseFN())

  private def buildContractFromDescriptor(): Future[BuildResult[BaseUnit]] = {
    val builder = APIContractUnitClientBuilder(dependencyFetcher)
    rl.foreach(builder.withResourceLoader)
    builder
      .build(projectDescriptor)
      .flatMap(_.build(withIdShortening = false))
      .map(BuildResult(_))
  }

  private def buildInstance(instance: Instance): Future[BuildResult[BaseUnit]] = {
    val client = APIInstanceClient(instance.gcl, instance.definedBy, rl)
    getCachedOrBuild(client.build)(instance.gcl)
  }

  private def buildInstances(): Future[Seq[BuildResult[BaseUnit]]] =
    Future.sequence(projectDescriptor.instances.map(buildInstance))

  private def buildContract(): Future[BuildResult[BaseUnit]] =
    getCachedOrBuild(buildContractFromDescriptor)(projectDescriptor.main)

  private def buildConfig = {
    APIConfiguration
      .API()
      .withRenderOptions(RenderOptions().withPrettyPrint.withCompactUris.withGovernanceMode)
      .withAliases(aliases)
  }
}

trait ProjectNodeCache {
  def fetch(uri: String): Option[Future[BaseUnit]]
}
