package org.mulesoft.apb.client.scala

import amf.apicontract.client.scala.APIConfiguration
import amf.apicontract.client.scala.model.domain.api.Api
import amf.core.client.scala.AMFResult
import amf.core.client.scala.config.RenderOptions
import amf.core.client.scala.model.document.{BaseUnit, Document}
import amf.core.client.scala.model.domain.{AmfObject, DomainElement}
import amf.core.client.scala.resource.ResourceLoader
import amf.core.internal.unsafe.PlatformSecrets
import amf.shapes.client.scala.config.JsonLDSchemaConfiguration
import amf.shapes.client.scala.model.domain.jsonldinstance.JsonLDObject
import org.mulesoft.apb.client.scala.extensions.APIProjectExtension
import org.mulesoft.apb.client.scala.model.{Project, ProjectBuilder}
import org.mulesoft.apb.internal.convert.ElementConverters.{AMFResultConverter, AmfObjectConverter}
import org.mulesoft.apb.internal.view.{ApiSummaryView, Views}
import org.mulesoft.apb.project.client.scala.environment.DependencyFetcher
import org.mulesoft.apb.project.client.scala.model.{Instance, ProjectDescriptor}
import org.mulesoft.apb.project.internal.descriptor.ApiProjectNamespaces
import org.mulesoft.apb.project.internal.descriptor.ApiProjectNamespaces.aliases

import scala.concurrent.ExecutionContext.Implicits.global
import scala.concurrent.Future
// TODO native-jsonld: change policies for a list of generic runtime dependencies with their own definition for merging
class APIProjectClient private[apb] (
    projectDescriptor: ProjectDescriptor,
    dependencyFetcher: DependencyFetcher,
    rl: Option[ResourceLoader],
    cache: Option[ProjectNodeCache],
    extensions: Seq[APIProjectExtension]
) extends PlatformSecrets {

  def build(): Future[Project] = {
    val builder = ProjectBuilder(projectDescriptor).withExtensions(extensions)
    for {
      _ <- buildContract().map(builder.withContract)
      _ <- buildInstances().map(i => builder.withInstances(i.toList))
    } yield builder.build()
  }

  def serialize(): Future[String] = {
    build().map(p => buildConfig.baseUnitClient().render(p.document))
  }

  def summary(schemaBase: String): Future[String] = {
    val summary = ApiSummaryView(schemaBase)
    build().map(project => summary.view(project.projectInfo))
  }

  private def getCachedOrBuild(orElseFN: () => Future[AmfObject])(uri: String): Future[AmfObject] =
    cache.flatMap(_.fetch(uri)).getOrElse(orElseFN())

  private def buildContractFromDescriptor(): Future[AmfObject] = {
    val builder = ApiContractUnitClientBuilder(dependencyFetcher)
    rl.foreach(builder.withResourceLoader)
    builder
      .build(projectDescriptor)
      .flatMap(_.build())
      .map(_.baseUnit)
  }

  private def buildInstance(instance: Instance): Future[JsonLDObject] = {
    val client = APIInstanceClient(instance.gcl, instance.definedBy, rl)
    getCachedOrBuild(client.build)(instance.gcl).map(_.toJsonLDObject())
  }

  private def buildInstances(): Future[Seq[JsonLDObject]] =
    Future.sequence(projectDescriptor.instances.map(buildInstance))

  private def buildContract(): Future[BaseUnit] =
    getCachedOrBuild(buildContractFromDescriptor)(projectDescriptor.main).map(_.toUnit())

  private def buildConfig = {
    APIConfiguration
      .API()
      .withRenderOptions(RenderOptions().withPrettyPrint.withCompactUris.withGovernanceMode)
      .withAliases(aliases)
  }
}

trait ProjectNodeCache {
  def fetch(uri: String): Option[Future[AmfObject]]
}
