package org.mulesoft.apb.client.scala

import amf.core.client.scala.model.domain.{AmfObject, DomainElement}
import amf.core.client.scala.parse.document.{ParsedDocument, ParserContext, SyamlParsedDocument}
import amf.core.client.scala.resource.ResourceLoader
import amf.core.internal.plugins.syntax.SyamlSyntaxParsePlugin
import amf.core.internal.unsafe.PlatformSecrets
import amf.custom.validation.internal.report.loaders.InMemoryResourceLoader
import amf.shapes.client.scala.config.JsonLDSchemaConfiguration
import amf.shapes.client.scala.model.document.{JsonLDInstanceDocument, JsonSchemaDocument}
import org.mulesoft.apb.internal.gcl.GclDefaults
import org.mulesoft.apb.internal.loaders.NestedDocumentRL
import org.mulesoft.apb.internal.plugins.GCLHackedSyntaxPlugin
import org.mulesoft.apb.project.client.scala.instances.APIInstanceBuilder
import org.yaml.model.YDocument
import org.yaml.parser.YamlParser

import scala.concurrent.ExecutionContext.Implicits.global
import scala.concurrent.Future

class APIInstanceClient private (instance: String, definedByUri: String, amfConfig: JsonLDSchemaConfiguration)
    extends PlatformSecrets {
  def build(): Future[AmfObject] = parseGCL().map(new APIInstanceBuilder(_).build())

  private def parseGCL(): Future[List[JsonLDInstanceDocument]] = {
    for {
      jsonSchema <- amfConfig.baseUnitClient().parseJsonLDSchema(definedByUri).map(_.jsonDocument)
      gclContent <- platform.fetchContent(instance, amfConfig)
      instances  <- parseGCLDocuments(gclContent.stream.toString, jsonSchema)
    } yield instances
  }

  private def parseGCLDocuments(
      gclContent: String,
      schema: JsonSchemaDocument
  ): Future[List[JsonLDInstanceDocument]] = {
    val docs   = parseDocumentsIn(gclContent)
    val client = amfConfig.withPlugin(GCLHackedSyntaxPlugin(docs)).withResourceLoader(NestedDocumentRL).baseUnitClient()
    Future.sequence(docs.keys.toList.map(client.parseJsonLDInstance(_, schema).map(_.instance)))
  }

  private def parseDocumentsIn(gclContent: String): Map[String, YDocument] = {
    YamlParser(gclContent)
      .documents()
      .zipWithIndex
      .map({ case (doc, i) => (s"file://gcl-instance/$i.yaml" -> doc) })
      .toMap
  }

  private case class GCLHackedSyntaxPlugin(index: Map[String, YDocument]) extends SyamlSyntaxParsePlugin {

    override def parse(text: CharSequence, mediaType: String, ctx: ParserContext): ParsedDocument = SyamlParsedDocument(
        index(text.toString)
    )
  }
}

private[apb] object APIInstanceClient {

  def build(instance: String, rl: ResourceLoader): Future[AmfObject] = apply(instance, None, Some(rl)).build()

  private[apb] def apply(
      instance: String,
      descriptedDefinedBy: Option[String],
      rl: Option[ResourceLoader]
  ): APIInstanceClient = {
    val (amfConfig, definedBy) = computeApiInstanceParameters(descriptedDefinedBy, rl)
    new APIInstanceClient(instance, definedBy, amfConfig)
  }

  private def computeApiInstanceParameters(descriptedDefinedBy: Option[String], rl: Option[ResourceLoader]) = {
    val baseConfig = jsonLdSchemaConfig(rl)
    descriptedDefinedBy match {
      case Some(db) => (baseConfig, db)
      case _        => GclDefaults(baseConfig)
    }
  }

  private def jsonLdSchemaConfig(rl: Option[ResourceLoader]) = {
    val base = JsonLDSchemaConfiguration.JsonLDSchema()
    rl.fold(base)(r => base.withResourceLoader(r))
  }
}
