package org.mulesoft.apb.client.scala

import amf.core.client.common.remote.Content
import amf.core.client.common.validation.SeverityLevels
import amf.core.client.scala.AMFGraphConfiguration
import amf.core.client.scala.model.document.ExternalFragment
import amf.core.client.scala.model.domain.ExternalDomainElement
import amf.core.client.scala.resource.ResourceLoader
import amf.core.client.scala.validation.AMFValidationResult
import amf.core.internal.remote.{FileNotFound, UnsupportedUrlScheme}
import amf.core.internal.unsafe.PlatformSecrets
import amf.shapes.client.scala.model.document.JsonLDInstanceDocument
import org.mulesoft.apb.project.client.scala.model.descriptor.documentation.{Documentation, DocumentationModel}
import org.mulesoft.apb.project.client.scala.model.project.documentation.APIDocumentation
import org.mulesoft.apb.project.internal.BaseUnitBuildResult
import org.mulesoft.apb.project.internal.idadoption.APBIdAdopter

import scala.concurrent.ExecutionContext.Implicits.global
import scala.concurrent.Future

class APIDocumentationClient(docNode: Documentation, rl: Option[ResourceLoader]) extends PlatformSecrets {

  val documentationPath: String = checkProtocol(docNode.path)

  def build(): Future[BaseUnitBuildResult] = fetch(docNode.path).map(buildAPIDocumentation)

  def buildAPIDocumentation(unitResult: BaseUnitBuildResult): BaseUnitBuildResult =
    APIDocumentationResultBuilder(unitResult).build()

  private val getConfig = {
    val configuration = AMFGraphConfiguration.predefined()
    rl.fold(configuration)(r => configuration.withResourceLoader(r))
  }

  private def fetch(path: String): Future[BaseUnitBuildResult] = {
    try {
      platform.fetchContent(path, getConfig).map(buildResult).recover {
        case e: FileNotFound => buildError(s"${e.getMessage}. Path: $path")
        case e: Throwable    => buildError(e.getMessage)
      }
    } catch {
      case unsupported: UnsupportedUrlScheme =>
        Future.successful(buildError(s"${unsupported.getMessage}'$path'"))
    }
  }

  private def buildResult(content: Content) = BaseUnitBuildResult(
      ExternalFragment()
        .withEncodes(
            ExternalDomainElement()
              .withRaw(content.stream.toString)
        )
  )

  private def buildError(message: String) = BaseUnitBuildResult(buildResult(message))

  private def buildResult(message: String) = AMFValidationResult(
      message,
      SeverityLevels.VIOLATION,
      docNode.wrapped.id,
      Some(DocumentationModel.Path.toString()),
      "error-building-documentation",
      None,
      None,
      null
  )

  private def checkProtocol(path: String) = {
    if (!path.startsWith("file://")) "file://" + path
    else path
  }

  private case class APIDocumentationResultBuilder(unitResult: BaseUnitBuildResult) {
    def build(): BaseUnitBuildResult = {
      unitResult.result match {
        case externalFragment: ExternalFragment => buildFromExternalFragment(externalFragment)
        case other => resultWithError(s"Unsupported unit type ${other.meta.typeIris.head} for documentation")
      }
    }

    private def resultWithError(error: String) = {
      unitResult.copy(report = unitResult.report.copy(results = (unitResult.report.results :+ buildResult(error))))
    }

    def buildFromExternalFragment(externalFragment: ExternalFragment): BaseUnitBuildResult = {
      Option(externalFragment.encodes).flatMap(_.raw.option()) match {
        case Some(value) => documentationToUnitResult(apiDocumentationFromRaw(value))
        case _           => documentationToUnitResult(apiDocumentation())
      }
    }

    def documentationToUnitResult(apiDocumentation: APIDocumentation): BaseUnitBuildResult = {
      unitResult.copy(result = jsonLDAdoptedDocument(apiDocumentation))
    }

    def jsonLDAdoptedDocument(apiDocumentation: APIDocumentation): JsonLDInstanceDocument = {
      val document = JsonLDInstanceDocument().withEncodes(Seq(apiDocumentation.wrapped))
      new APBIdAdopter(documentationPath).adoptFromRoot(document)
      document
    }

    def apiDocumentationFromRaw(raw: String): APIDocumentation = apiDocumentation().withContent(raw)

    def apiDocumentation(): APIDocumentation = {
      val apiDocumentation = APIDocumentation().withPath(docNode.path)
      docNode.name.foreach(apiDocumentation.withName)
      apiDocumentation
    }
  }

}
