package org.mulesoft.apb.client.scala

import amf.apicontract.client.scala.APIConfiguration
import amf.core.client.common.validation.{ProfileName, ProfileNames, SeverityLevels, UnknownProfile}
import amf.core.client.scala.config.RenderOptions
import amf.core.client.scala.model.document.{BaseUnit, Document}
import amf.core.client.scala.resource.ResourceLoader
import amf.core.client.scala.validation.{AMFValidationReport, AMFValidationResult}
import amf.core.internal.unsafe.PlatformSecrets
import amf.shapes.client.scala.model.domain.jsonldinstance.JsonLDObject
import org.mulesoft.apb.client.scala.contract.{APIContractClient, APIContractClientBuilder}
import org.mulesoft.apb.client.scala.instances.APIInstanceClient
import org.mulesoft.apb.internal.convert.ElementConverters.AmfObjectConverter
import org.mulesoft.apb.internal.lint.APIProjectLinter
import org.mulesoft.apb.project.client.scala.ProjectConfiguration
import org.mulesoft.apb.project.client.scala.environment.DependencyFetcher
import org.mulesoft.apb.project.client.scala.extensions.APIProjectExtension
import org.mulesoft.apb.project.client.scala.model.{BaseUnitBuildResult, ProjectBuildResult}
import org.mulesoft.apb.project.client.scala.model.descriptor.{Gav, Instance}
import org.mulesoft.apb.project.client.scala.model.descriptor.documentation.Documentation
import org.mulesoft.apb.project.client.scala.model.project.{Project, ProjectDocumentBuilder}
import org.mulesoft.apb.project.internal.descriptor.ApiProjectNamespaces.aliases
import org.mulesoft.apb.project.internal.gcl.SchemaProvider
import org.mulesoft.apb.project.internal.idadoption.APBIdAdopter
import org.mulesoft.apb.project.internal.idadoption.URITools._
import org.mulesoft.apb.project.internal.instances.{ExtensionAssetParser, ScopedExtensionIndex}
import org.mulesoft.apb.project.internal.view.ApiSummaryView

import scala.concurrent.ExecutionContext.Implicits.global
import scala.concurrent.Future
// TODO native-jsonld: change policies for a list of generic runtime dependencies with their own definition for merging
class APIProjectClient private[apb] (
    projectConfiguration: ProjectConfiguration,
    dependencyFetcher: DependencyFetcher,
    resourceLoaders: List[ResourceLoader],
    cache: Option[ProjectNodeCache],
    extensions: Seq[APIProjectExtension],
    base: Option[String] = None
) extends PlatformSecrets {

  def assemble(): Future[ProjectBuildResult] = {
    val builder = ProjectDocumentBuilder(projectConfiguration).withExtensions(extensions)
    for {
      contractOption <- buildContract()
      builtInstances <- buildInstances()
    } yield {
      val (instances, instancesError) = extractJsonLDNodes(builtInstances)
      contractOption.foreach(co => builder.withContract(co.result, co.report.results))
      builder.withInstances(instances, instancesError)
      val documentations = buildDocumentation()
      builder.withDocumentation(documentations, Nil)
      base.foreach(builder.withBase)
      builder.build()
    }
  }

  def validate(project: Project): Future[AMFValidationReport] = {
    val apiProfile = ProfileName("API")
    projectConfiguration.descriptor.main
      .map(new APIContractClient(projectConfiguration, _))
      .map(_.validate(project.apiContract()).map(_.copy(profile = apiProfile)))
      .getOrElse(Future.successful(AMFValidationReport.empty("", apiProfile)))
    // instance validation?
  }

  def validate(): Future[AMFValidationReport] = {
    for {
      ass              <- assemble()
      validationReport <- validate(ass.result)
    } yield ass.report.merge(validationReport)
  }

  def serialize(): Future[String] = {
    assemble().map(p => buildConfig.baseUnitClient().render(p.result.document))
  }

  def serialize(project: Project): String = buildConfig.baseUnitClient().render(project.document)

  def summary(schemaBase: String): Future[String] = {
    val summary = ApiSummaryView(schemaBase)
    assemble().map(project => summary.view(project.result.projectInfo))
  }

  def summary(schemaBase: String, project: Project): String = {
    val summary = ApiSummaryView(schemaBase)
    summary.view(project.projectInfo)
  }

  def lint(rulesets: List[Gav]): Future[List[AMFValidationReport]] = {
    assemble().flatMap(result => lint(rulesets, result.result))
  }

  def lint(rulesets: List[Gav], project: Project): Future[List[AMFValidationReport]] = {
    val graph = serialize(project)
    projectLinter(graph).lint(rulesets)
  }
  def lint(): Future[List[AMFValidationReport]] = {
    serialize().flatMap(projectLinter(_).lintProfiles(projectConfiguration.profileDependencies.map(_.profile).toList))
  }

  def lint(project: Project): Future[List[AMFValidationReport]] = {
    val graph = serialize(project)
    projectLinter(graph).lintProfiles(projectConfiguration.profileDependencies.map(_.profile).toList)
  }

  private def projectLinter(graph: String) = new APIProjectLinter(dependencyFetcher, graph)
  private def extractJsonLDNodes(units: Seq[BaseUnitBuildResult]) =
    (units.flatMap(_.result.toJsonLDObjects()).toList, units.flatMap(_.report.results))

  private def getCachedOrBuild(orElseFN: () => Future[BaseUnitBuildResult])(uri: String): Future[BaseUnitBuildResult] =
    cache
      .flatMap(_.fetch(uri))
      .map(future => future.map(unit => BaseUnitBuildResult(unit)))
      .getOrElse(orElseFN())

  private def buildContractFromDescriptor(): Future[BaseUnitBuildResult] = {
    val builder = APIContractClientBuilder(dependencyFetcher)
    builder.withResourceLoaders(resourceLoaders)
    builder
      .build(projectConfiguration)
      .build(withIdShortening = false)
      .map(BaseUnitBuildResult(_))
  }

  private def buildInstance(instance: Instance, index: ScopedExtensionIndex): Future[BaseUnitBuildResult] = {
    val client = APIInstanceClient(instance, resourceLoaders, index)
    getCachedOrBuild(client.build)(instance.gcl)
  }

  private def buildInstances(): Future[Seq[BaseUnitBuildResult]] = {
    for {
      extensionSchema <- SchemaProvider.extensionSchema
      extensionIndex <- ExtensionAssetParser(dependencyFetcher, extensionSchema)
        .parse(
            projectConfiguration.descriptor
        ) // TODO: move parsing to project configuration building (are fixed, cannot change)
        .map(_.scoped(projectConfiguration.descriptor.dependencies().map(_.gav).toSet))
      instances <- Future.sequence(
          projectConfiguration.descriptor.instances.map(instance => buildInstance(instance, extensionIndex))
      )
    } yield {
      instances
    }
  }

  private def buildContract(): Future[Option[BaseUnitBuildResult]] = {
    projectConfiguration.descriptor.main match {
      case Some(main) => getCachedOrBuild(buildContractFromDescriptor)(main).map(Some(_))
      case _          => Future.successful(None)
    }
  }

  private def buildDocumentation(): List[JsonLDObject] =
    projectConfiguration.descriptor.documentation.map(doc => buildDocumentationNode(doc)).toList

  private def buildDocumentationNode(docNode: Documentation): JsonLDObject = {
    new APBIdAdopter(docNode.path.fromPath).adoptFromRelative(docNode.wrapped)
    docNode.wrapped
  }
  private def buildConfig = {
    APIConfiguration
      .API()
      .withRenderOptions(RenderOptions().withPrettyPrint.withCompactUris.withGovernanceMode)
      .withAliases(aliases)
  }
}

trait ProjectNodeCache {
  def fetch(uri: String): Option[Future[BaseUnit]]
}

object NoContractMainBuildResult {
  val errorDoc: Document = Document().withId("amf://error")
  val validationResult: AMFValidationResult =
    AMFValidationResult("Missing main file", SeverityLevels.VIOLATION, errorDoc, None, "", None, None, null)
  val result: BaseUnitBuildResult =
    BaseUnitBuildResult(errorDoc, AMFValidationReport(errorDoc.id, UnknownProfile, Seq(validationResult)))

}
