package org.mulesoft.apb.client.scala.instances

import amf.core.client.common.validation.{ProfileName, ProfileNames, SeverityLevels, StrictValidationMode}
import amf.core.client.scala.model.domain.{AmfArray, Shape}
import amf.core.client.scala.parse.document.{ParsedDocument, ParserContext, SyamlParsedDocument}
import amf.core.client.scala.resource.ResourceLoader
import amf.core.client.scala.validation.{AMFValidationReport, AMFValidationResult}
import amf.core.internal.parser.YMapOps
import amf.core.internal.plugins.syntax.SyamlSyntaxParsePlugin
import amf.core.internal.remote.Mimes
import amf.core.internal.unsafe.PlatformSecrets
import amf.shapes.client.scala.config.{JsonLDSchemaConfiguration, JsonSchemaConfiguration}
import amf.shapes.client.scala.model.document.{JsonLDInstanceDocument, JsonSchemaDocument}
import amf.shapes.client.scala.model.domain.jsonldinstance.JsonLDObject
import amf.shapes.internal.document.metamodel.JsonLDInstanceDocumentModel.Encodes
import org.mulesoft.apb.internal.gcl.GclDefaults
import org.mulesoft.apb.internal.loaders.NestedDocumentRL
import org.mulesoft.apb.project.client.scala.instances.APIInstanceBuilder
import org.mulesoft.apb.project.client.scala.model.{BaseUnitBuildResult, JsonLDInstanceDocumentBuildResult}
import org.mulesoft.apb.project.client.scala.model.DynamicObject.toJsonLDObject
import org.mulesoft.apb.project.client.scala.model.descriptor.Instance
import org.mulesoft.apb.project.internal.idadoption.IdAdopterProvider
import org.mulesoft.apb.project.internal.idadoption.URITools._
import org.mulesoft.apb.project.internal.instances.ResourceKind.Other
import org.mulesoft.apb.project.internal.instances.{ErrorAPIInstance, ResourceKind, ScopedExtensionIndex}
import org.mulesoft.common.collections.Group
import org.yaml.model.{YDocument, YMap, YNode}
import org.yaml.parser.YamlParser
import org.yaml.render.{JsonRender, JsonRenderOptions}

import scala.collection.mutable.ArrayBuffer
import scala.concurrent.ExecutionContext.Implicits.global
import scala.concurrent.Future

class APIInstanceClient private (
    instance: Instance,
    amfConfig: JsonLDSchemaConfiguration,
    extensionIndex: ScopedExtensionIndex
) extends PlatformSecrets {
  def build(): Future[BaseUnitBuildResult] = {
    parseGcls().map(resources => buildResultFromResources(resources))
  }
  private def buildResultFromResources(resources: List[JsonLDInstanceDocumentBuildResult]): BaseUnitBuildResult = {
    BaseUnitBuildResult(assembleInstances(resources, extensionIndex), mergeReports(resources))
  }

  private def parseGcls(): Future[List[JsonLDInstanceDocumentBuildResult]] = {
    for {
      gclContent <- platform.fetchContent(instance.gcl, amfConfig)
      instances  <- parseDocuments(gclContent.stream.toString)
    } yield instances
  }

  private[apb] def parseDocuments(gclContent: String): Future[List[JsonLDInstanceDocumentBuildResult]] = {
    val docs    = parseDocumentsIn(gclContent)
    val grouped = groupGclByKind(docs)
    val instances = grouped.map { case (kind, docs) =>
      checkAndParseDocsByResource(kind, docs)
    }
    Future.sequence(instances).map(_.flatten.toList)
  }

  private def checkAndParseDocsByResource(kind: String, docs: ArrayBuffer[YDocument]) = {
    if (ResourceKind.ValidKinds.exists(_.kind == kind)) parseGclResources(kind, docs.toList)
    else Future.successful(buildInvalidKindResult(kind, docs).toList)
  }
  private def buildInvalidKindResult(kind: String, docs: ArrayBuffer[YDocument]) = {
    docs.map { doc =>
      val errorDocument = JsonLDInstanceDocument().withId(instance.gcl.withProtocol)
      JsonLDInstanceDocumentBuildResult(
          errorDocument,
          AMFValidationReport("", ProfileName("GCL"), Seq(buildKindErrorFor(errorDocument, kind)))
      )
    }
  }

  private def buildKindErrorFor(node: JsonLDInstanceDocument, kind: String) = {
    AMFValidationResult(
        s"Invalid kind: $kind",
        SeverityLevels.VIOLATION,
        node,
        None,
        "invalid-kind",
        None,
        Some(instance.gcl),
        null
    )
  }

  private def parseGclResources(kind: String, docs: List[YDocument]) = {
    for {
      schema    <- amfConfig.baseUnitClient().parseJsonLDSchema(kind).map(_.jsonDocument)
      instances <- parseInstances(docs, schema, kind)
    } yield {
      instances
    }
  }

  private def groupGclByKind(docs: List[YDocument]) = {
    Group(docs).legacyGroupBy(doc => GclResource(doc.node).kind.getOrElse(Other.kind))
  }

  private def parseDocumentsIn(gclContent: String): List[YDocument] = {
    YamlParser(gclContent).documents().toList
  }

  private def parseInstances(docs: List[YDocument], schema: JsonSchemaDocument, kind: String) = {
    val docIndex = buildGclUriIndex(docs, kind)
    val client   = buildClient(docIndex)
    val results = docIndex.map { case (_, documentWithUri) =>
      for {
        // we send to AMF the path uri without encoding it, because AMF will always encode it
        instance <- client.parseJsonLDInstance(documentWithUri.unencodedUri, schema).map(_.instance)
      } yield {
        val report = validate(documentWithUri.document, schema.encodes)
        JsonLDInstanceDocumentBuildResult(instance, report)
      }
    }
    Future.sequence(results).map(_.toList)
  }

  private val formattedInstancePath = this.instance.gcl.withProtocol

  private case class DocumentWithUri(unencodedUri: String, document: YDocument)

  private def buildGclUriIndex(docs: List[YDocument], kind: String): Map[String, DocumentWithUri] = {
    toGCLUriList(docs, kind)
      // we encode the path uri to match what AMF will request
      .map(index => (platform.encodeURI(index._1), DocumentWithUri(index._1, index._2)))
      .toMap
  }

  private def toGCLUriList(docs: List[YDocument], kind: String): List[(String, YDocument)] = {
    if (ResourceKind.ApiInstance.kind == kind) docs.map(d => formattedInstancePath -> d)
    else if (docs.size < 2) docs.map(d => formattedInstancePath.forKind(kind) -> d)
    else docs.zipWithIndex.map { case (d, index) => formattedInstancePath.indexed(index, kind) -> d }
  }

  private def buildClient(docs: Map[String, DocumentWithUri]) = {
    val syntaxPlugin = GCLHackedSyntaxPlugin(docs)
    amfConfig
      .withPlugin(syntaxPlugin)
      .withResourceLoader(NestedDocumentRL)
      .withIdAdopterProvider(IdAdopterProvider)
      .baseUnitClient()
  }

  private def validate(doc: YDocument, schema: Shape): AMFValidationReport = {
    val json = JsonRender.render(doc, 0, JsonRenderOptions().withoutNonAsciiEncode)
    validate(schema, json)
  }

  private def validate(schema: Shape, payload: String): AMFValidationReport = {
    val client = JsonSchemaConfiguration.JsonSchema().elementClient()
    client
      .payloadValidatorFor(schema, Mimes.`application/json`, StrictValidationMode)
      .syncValidate(payload)
  }

  private def mergeReports(resources: List[JsonLDInstanceDocumentBuildResult]) = {
    resources.map(_.report).foldLeft(AMFValidationReport.empty("", ProfileNames.AMF)) { (acc, curr) => acc.merge(curr) }
  }

  private def assembleInstances(result: List[JsonLDInstanceDocumentBuildResult], index: ScopedExtensionIndex) = {
    val instances = new APIInstanceBuilder(result.map(_.result), index).build()
    if (instances.isEmpty) wrap(List(ErrorAPIInstance))
    else wrap(instances.map(toJsonLDObject(_)))
  }

  private def wrap(nodes: Seq[JsonLDObject]) = JsonLDInstanceDocument().setWithoutId(Encodes, AmfArray(nodes))

  case class GclResource(node: YNode) {

    lazy val map: YMap = node.as[YMap]

    def kind: Option[String] = map.key("kind").map(_.value.as[String])
  }

  private case class GCLHackedSyntaxPlugin(index: Map[String, DocumentWithUri]) extends SyamlSyntaxParsePlugin {
    override def parse(text: CharSequence, mediaType: String, ctx: ParserContext): ParsedDocument =
      SyamlParsedDocument(index(text.toString).document)
  }
}

private[apb] object APIInstanceClient extends PlatformSecrets {

  def build(instanceIri: Instance, resourceLoaders: List[ResourceLoader]): Future[BaseUnitBuildResult] =
    apply(instanceIri, resourceLoaders, ScopedExtensionIndex.Empty).build()

  def build(instanceIri: String): Future[BaseUnitBuildResult] = {
    apply(Instance(instanceIri), platform.loaders().toList, ScopedExtensionIndex.Empty).build()
  }

  private[apb] def apply(
      instance: Instance,
      resourceLoaders: List[ResourceLoader],
      index: ScopedExtensionIndex
  ): APIInstanceClient = {
    val amfConfig = computeApiInstanceParameters(instance.definedBy, resourceLoaders)
    new APIInstanceClient(instance, amfConfig, index.addScope(instance.dependencies.map(_.gav).toSet))
  }

  private def computeApiInstanceParameters(definedBy: Option[String], resourceLoaders: List[ResourceLoader]) = {
    val baseConfig = jsonLdSchemaConfig(resourceLoaders)
    definedBy match {
      case Some(_) => baseConfig
      case _       => GclDefaults(baseConfig)
    }
  }

  private def jsonLdSchemaConfig(resourceLoaders: List[ResourceLoader]) = {
    val base = JsonLDSchemaConfiguration.JsonLDSchema()
    base.withResourceLoaders(resourceLoaders)
  }
}
