package org.mulesoft.apb.internal.client.project

import amf.apicontract.client.scala.APIConfiguration
import amf.core.client.common.validation.ProfileName
import amf.core.client.scala.config.RenderOptions
import amf.core.client.scala.resource.ResourceLoader
import amf.core.client.scala.validation.AMFValidationReport
import amf.core.internal.unsafe.PlatformSecrets
import amf.shapes.client.scala.model.domain.jsonldinstance.JsonLDObject
import org.mulesoft.apb.client.scala.{APIProjectClient, ProjectNodeCache}
import org.mulesoft.apb.internal.client.contract.{APIContractClient, APIContractClientBuilder}
import org.mulesoft.apb.internal.client.instances.APIInstanceClient
import org.mulesoft.apb.internal.convert.ElementConverters.AmfObjectConverter
import org.mulesoft.apb.internal.lint.APIProjectLinter
import org.mulesoft.apb.project.client.scala.descriptor.DescriptorParseResult
import org.mulesoft.apb.project.client.scala.ProjectConfiguration
import org.mulesoft.apb.project.client.scala.environment.DependencyFetcher
import org.mulesoft.apb.project.client.scala.extensions.APIProjectExtension
import org.mulesoft.apb.project.client.scala.model.descriptor.documentation.Documentation
import org.mulesoft.apb.project.client.scala.model.descriptor.{Gav, Instance, ProjectDescriptor}
import org.mulesoft.apb.project.client.scala.model.project.{Project}
import org.mulesoft.apb.project.client.scala.model.{BaseUnitBuildResult, ProjectBuildResult}
import org.mulesoft.apb.project.internal.dependency.ProjectConfigurationBuilder
import org.mulesoft.apb.project.internal.descriptor.ApiProjectNamespaces.aliases
import org.mulesoft.apb.project.internal.gcl.SchemaProvider
import org.mulesoft.apb.project.internal.idadoption.APBIdAdopter
import org.mulesoft.apb.project.internal.idadoption.URITools.URIStr
import org.mulesoft.apb.project.internal.instances.{ExtensionAssetParser, ScopedExtensionIndex}
import org.mulesoft.apb.project.internal.model.project.ProjectDocumentBuilder
import org.mulesoft.apb.project.internal.view.ApiSummaryView

import scala.concurrent.ExecutionContext.Implicits.global
import scala.concurrent.Future
import scala.util.Success

// TODO native-jsonld: change policies for a list of generic runtime dependencies with their own definition for merging
class DefaultAPIProjectClient private[apb] (
    descriptor: DescriptorParseResult,
    dependencyFetcher: DependencyFetcher,
    resourceLoaders: List[ResourceLoader],
    cache: Option[ProjectNodeCache],
    extensions: Seq[APIProjectExtension],
    base: Option[String] = None
) extends APIProjectClient
    with PlatformSecrets {

  private var _project: Option[ProjectBuildResult]        = None
  private var _dependencies: Option[ProjectConfiguration] = None

  override def dependencies(): Future[ProjectConfiguration] = memoizedDependencies { () =>
    new ProjectConfigurationBuilder(dependencyFetcher, resourceLoaders)
      .build(descriptor)
  }

  def project(): Future[ProjectBuildResult] = memoizeProject { () =>
    dependencies().flatMap(project)
  }

  private def project(dependencies: ProjectConfiguration): Future[ProjectBuildResult] = memoizeProject { () =>
    val builder = ProjectDocumentBuilder(dependencies).withExtensions(extensions)
    for {
      contractOption <- buildContract(dependencies)
      builtInstances <- buildInstances(descriptor.descriptor)
    } yield {
      val (instances, instancesError) = extractJsonLDNodes(builtInstances)
      contractOption.foreach(co => builder.withContract(co.result, co.report.results))
      builder.withInstances(instances, instancesError)
      val documentations = buildDocumentation(descriptor.descriptor)
      builder.withDocumentation(documentations, Nil)
      base.foreach(builder.withBase)
      builder.build()
    }
  }

  def validate(): Future[AMFValidationReport] = {
    for {
      deps             <- dependencies()
      model            <- project()
      validationReport <- validate(model.result, deps)
    } yield model.report.merge(validationReport)
  }

  private def validate(project: Project, dependencies: ProjectConfiguration): Future[AMFValidationReport] = {
    val apiProfile = ProfileName("API")
    dependencies.descriptor.main
      .map(new APIContractClient(dependencies, _))
      .map(_.validate(project.apiContract()).map(_.copy(profile = apiProfile)))
      .getOrElse(Future.successful(AMFValidationReport.empty("", apiProfile)))
    // instance validation?
  }

  def serialize(): Future[String] = {
    for {
      model <- project()
    } yield {
      buildConfig.baseUnitClient().render(model.result.document)
    }
  }

  private def serialize(project: Project): String = buildConfig.baseUnitClient().render(project.document)

  def summary(schemaBase: String): Future[String] = {
    for {
      model <- project()
    } yield {
      summary(schemaBase, model.result)
    }
  }

  private def summary(schemaBase: String, project: Project): String = {
    val summary = ApiSummaryView(schemaBase)
    summary.view(project.projectInfo)
  }

  def lint(): Future[List[AMFValidationReport]] = {
    for {
      deps    <- dependencies()
      model   <- serialize()
      reports <- projectLinter(model).lintProfiles(deps.profileDependencies.map(_.profile).toList)
    } yield {
      reports
    }
  }

  def lint(rulesets: List[Gav]): Future[List[AMFValidationReport]] = {
    for {
      deps    <- dependencies()
      project <- project(deps)
      reports <- lint(rulesets, project.result)
    } yield {
      reports
    }
  }

  private def lint(rulesets: List[Gav], project: Project): Future[List[AMFValidationReport]] = {
    val graph = serialize(project)
    projectLinter(graph).lint(rulesets)
  }

  private def projectLinter(graph: String) = new APIProjectLinter(dependencyFetcher, graph)

  private def extractJsonLDNodes(units: Seq[BaseUnitBuildResult]) =
    (units.flatMap(_.result.toJsonLDObjects()).toList, units.flatMap(_.report.results))

  private def getCachedOrBuild(orElseFN: () => Future[BaseUnitBuildResult])(uri: String): Future[BaseUnitBuildResult] =
    cache
      .flatMap(_.fetch(uri))
      .map(future => future.map(unit => BaseUnitBuildResult(unit)))
      .getOrElse(orElseFN())

  private def buildContractFromDescriptor(dependencies: ProjectConfiguration)(): Future[BaseUnitBuildResult] = {
    val builder = APIContractClientBuilder(dependencyFetcher)
    builder.withResourceLoaders(resourceLoaders)
    builder
      .build(dependencies)
      .build(withIdShortening = false)
      .map(BaseUnitBuildResult(_))
  }

  private def buildInstance(instance: Instance, index: ScopedExtensionIndex): Future[BaseUnitBuildResult] = {
    val client = APIInstanceClient(instance, resourceLoaders, index)
    getCachedOrBuild(client.build)(instance.gcl)
  }

  private def buildInstances(descriptor: ProjectDescriptor): Future[Seq[BaseUnitBuildResult]] = {
    for {
      extensionSchema <- SchemaProvider.extensionSchema
      extensionIndex <- ExtensionAssetParser(dependencyFetcher, extensionSchema)
        .parse(
            descriptor
        ) // TODO: move parsing to project configuration building (are fixed, cannot change)
        .map(_.scoped(descriptor.dependencies().map(_.gav).toSet))
      instances <- Future.sequence(
          descriptor.instances.map(instance => buildInstance(instance, extensionIndex))
      )
    } yield {
      instances
    }
  }

  private def buildContract(dependencies: ProjectConfiguration): Future[Option[BaseUnitBuildResult]] = {
    dependencies.descriptor.main match {
      case Some(main) => getCachedOrBuild(buildContractFromDescriptor(dependencies))(main).map(Some(_))
      case _          => Future.successful(None)
    }
  }

  private def buildDocumentation(descriptor: ProjectDescriptor): List[JsonLDObject] =
    descriptor.documentation.map(doc => buildDocumentationNode(doc)).toList

  private def buildDocumentationNode(docNode: Documentation): JsonLDObject = {
    new APBIdAdopter(docNode.path.fromPath).adoptFromRelative(docNode.wrapped)
    docNode.wrapped
  }

  private def buildConfig = {
    APIConfiguration
      .API()
      .withRenderOptions(RenderOptions().withPrettyPrint.withCompactUris.withGovernanceMode)
      .withAliases(aliases)
  }

  private def memoizeProject(provider: () => Future[ProjectBuildResult]): Future[ProjectBuildResult] = {
    memoized(_project, provider, cacheProject)
  }

  private def memoizedDependencies(provider: () => Future[ProjectConfiguration]): Future[ProjectConfiguration] = {
    memoized(_dependencies, provider, cacheDependencies)
  }

  private def memoized[T](memoizedValue: Option[T], provider: () => Future[T], memoizer: T => Unit): Future[T] = {
    memoizedValue match {
      case Some(project) => Future.successful(project)
      case None =>
        provider().andThen { case Success(project) =>
          memoizer(project)
        }
    }
  }

  private def cacheProject(result: ProjectBuildResult): Unit = {
    if (result.report.conforms) {
      this._project = Some(result)
    }
  }

  private def cacheDependencies(dependencies: ProjectConfiguration): Unit = {
    this._dependencies = Some(
        dependencies
    ) // TODO: question, do we always cache although we have errors in dependency fetching ???
  }
}
