package org.mulesoft.apb.internal.lint

import amf.aml.client.scala.model.document.DialectInstance
import amf.core.client.scala.resource.ResourceLoader
import amf.core.client.scala.validation.AMFValidationReport
import amf.core.internal.unsafe.PlatformSecrets
import amf.custom.validation.client.ProfileValidatorNodeBuilder
import amf.custom.validation.client.scala.BaseProfileValidatorBuilder
import org.mulesoft.apb.internal.client.contract.APIContractClientBuilder
import org.mulesoft.apb.project.client.scala.environment.{DependencyFetcher, UnreachableGavException}
import org.mulesoft.apb.project.client.scala.model.descriptor.Gav

import scala.collection.immutable
import scala.concurrent.ExecutionContext.Implicits.global
import scala.concurrent.Future

class APIProjectLinter private[apb] (
    dependencyFetcher: DependencyFetcher,
    serializedProject: String,
    validatorBuilder: BaseProfileValidatorBuilder = ProfileValidatorNodeBuilder
) extends PlatformSecrets {

  def lint(rulesets: List[Gav]): Future[List[AMFValidationReport]] = {
    checkAccess(rulesets)
    Future.sequence(lintAll(rulesets))
  }

  private[apb] def lintProfiles(profiles: List[DialectInstance]): Future[List[AMFValidationReport]] =
    Future.sequence(profiles.map(report))

  private def checkAccess(rulesets: List[Gav]) = {
    val unreachable: immutable.Seq[Gav] =
      rulesets.filterNot(gav => dependencyFetcher.accepts(gav.groupId, gav.assetId, gav.version))
    if (unreachable.nonEmpty) {
      val messagePrefix   = "Unreachable gavs: "
      val message: String = unreachable.foldLeft(messagePrefix)((concatMessage, gav) => concatMessage + gav.path + ", ")
      throw UnreachableGavException(message.stripSuffix(", "))
    }
  }

  private def lintAll(rulesets: List[Gav]) = rulesets.map(lint)

  private def lint(gav: Gav) = resolve(gav).flatMap(report)

  private def resolve(gav: Gav) = {
    val dependencyLoader = fetch(gav)
    buildRuleset(dependencyLoader)
  }

  private def report(profile: DialectInstance) = {
    validatorBuilder
      .validator(profile)
      .validate(serializedProject, profile.location().getOrElse(""))
  }

  private def fetch(gav: Gav) = dependencyFetcher.fetch(gav.groupId, gav.assetId, gav.version)

  private def buildRuleset(loader: ResourceLoader) = {
    APIContractClientBuilder(dependencyFetcher)
      .withResourceLoader(loader)
      .build("file://")
      .flatMap(_.build().map(_.baseUnit))
      .collect({ case d: DialectInstance =>
        d // what if not a dialect instance defined by a ValidationProfile? should we throw an exception?
      })
  }

}
