package org.mulesoft.apb.client.scala.extensions

import amf.core.client.scala.model.StrField
import amf.core.client.scala.model.domain.{AmfObject, AmfScalar, DomainElement}
import amf.core.client.scala.vocabulary.ValueType
import amf.core.internal.metamodel.Type.Str
import amf.core.internal.metamodel.{Field, ModelDefaultBuilder, Obj}
import amf.core.internal.parser.domain.{Annotations, Fields}
import amf.shapes.client.scala.model.domain.jsonldinstance.JsonLDObject
import org.mulesoft.apb.client.scala.extensions.EnvironmentExtension.ENVIRONMENT_IRI
import org.mulesoft.apb.client.scala.extensions.InstanceEnvironmentModel.{Id, Name}
import org.mulesoft.apb.client.scala.model.Project
import org.mulesoft.apb.client.scala.model.ProjectBuilder.INSTANCE_ID_IRI
import org.mulesoft.apb.internal.metamodel.ProjectInfoModel.Instances
import org.mulesoft.apb.project.client.scala.gcl.ApiInstance
import org.mulesoft.apb.project.internal.descriptor.ApiProjectNamespaces.{Anypoint, Core, Management}
import org.mulesoft.apb.project.internal.ops.JsonLdObjectOps

import scala.collection.mutable

trait APIProjectExtension {

  def extend(project: Project): Project
}

object EnvironmentExtension {
  val ENVIRONMENT_IRI = Field(InstanceEnvironmentModel, Management + "environment")
}

case class EnvironmentExtension(envs: Seq[InstanceEnvironment]) extends APIProjectExtension {

  private val grouped = envs.collect {
    case env if env.environmentId().nonEmpty => env.environmentId().value() -> env
  }.toMap

  override def extend(project: Project): Project = {
    val next = project.instances().map(setInstanceId).map(setEnvironment)
    project.projectInfo.setArray(Instances, next)
    project
  }

  private def setEnvironment(instance: JsonLDObject) = {
    val query = ApiInstance(instance)
    val enhanced = query.environmentId.flatMap(grouped.get).map { env =>
      val clonedEnv: AmfObject = env.cloneElement(mutable.Map.empty)
      JsonLdObjectOps.addProperty(instance, ENVIRONMENT_IRI, clonedEnv)
    }
    enhanced.getOrElse(instance)
  }

  private def setInstanceId(instance: JsonLDObject) = {
    ApiInstance(instance).instanceId
      .map(id => JsonLdObjectOps.addProperty(instance, INSTANCE_ID_IRI, AmfScalar(id)))
      .getOrElse(instance)
  }
}

case class InstanceEnvironment private[apb] (fields: Fields, annotations: Annotations) extends DomainElement {
  override def meta: Obj = InstanceEnvironmentModel

  def environmentId(): StrField = fields.field(Id)
  def name(): StrField          = fields.field(Name)
  def `type`(): StrField        = fields.field(InstanceEnvironmentModel.Type)

  def withEnvironmentId(value: String) = set(Id, value)
  def withName(value: String)          = set(Name, value)
  def withType(value: String)          = set(InstanceEnvironmentModel.Type, value)

  override def componentId: String = environmentId().option().getOrElse("environment")
}

object InstanceEnvironment {
  def apply(): InstanceEnvironment = InstanceEnvironment(Fields(), Annotations())
}

object InstanceEnvironmentModel extends ModelDefaultBuilder {

  val Id = Field(
      Str,
      Anypoint + "environmentID"
  )

  val Name = Field(
      Str,
      Core + "name"
  )

  val Type = Field(
      Str,
      Anypoint + "environmentType"
  )

  override def modelInstance: AmfObject = InstanceEnvironment()

  override def fields: List[Field] = List(Id, Name, Type)

  override val `type`: scala.List[ValueType] = List(ValueType(Management, "Environment"))
}
