package org.mulesoft.apb.internal.ast

import org.mulesoft.common.client.lexical.SourceLocation
import org.yaml.model.{YMap, YMapEntry, YNode, YType}

object JsonMergePatch {

  def merge(target: YNode, patch: YNode): YNode = (target.value, patch.value) match {
    case (targetMap: YMap, patchMap: YMap) =>
      applyPatch(target, targetMap, patchMap)
    case (_, patchMap: YMap) =>
      applyPatch(target, YMap.empty, patchMap)
    case _ => patch
  }

  private def applyPatch(target: YNode, targetMap: YMap, patchMap: YMap): YNode = {
    val starter               = (IndexedSeq.empty[YMapEntry], patchMap.map)
    val (entries, notApplied) = applyPatch(targetMap, starter)
    val result                = applyPatchNotInTarget(entries, patchMap, notApplied)
    YNode(YMap(target.location, result))
  }

  private def applyPatch(
      targetMap: YMap,
      starter: (IndexedSeq[YMapEntry], Map[YNode, YNode])
  ): (IndexedSeq[YMapEntry], Map[YNode, YNode]) = {
    targetMap.entries.foldLeft(starter) { (acc, entry) =>
      val (entries, patch) = acc
      applyPatch(entries, entry, patch)
    }
  }

  private def applyPatch(
      entries: IndexedSeq[YMapEntry],
      entry: YMapEntry,
      patch: Map[YNode, YNode]
  ): (IndexedSeq[YMapEntry], Map[YNode, YNode]) = {
    patch.get(entry.key) match {
      case Some(targetValue) if isNullValue(targetValue) => (entries, patch - entry.key)
      case Some(targetValue) =>
        val merged = copy(entry, merge(entry.value, targetValue))
        (entries :+ merged, patch - entry.key)
      case None => (entries :+ entry, patch)
    }
  }

  private def applyPatchNotInTarget(entries: IndexedSeq[YMapEntry], patch: YMap, notApplied: Map[YNode, YNode]) = {
    val patchEntries = patch.entries.filter(entry => notApplied.contains(entry.key))
    val result       = deepFilterOutNulls(patchEntries) ++ entries
    result
  }

  private def deepFilterOutNulls(patchEntries: IndexedSeq[YMapEntry]): IndexedSeq[YMapEntry] = {
    patchEntries.foldLeft(IndexedSeq.empty[YMapEntry]) { (acc, entry) => deepFilterOutNulls(acc, entry) }
  }

  private def isNullValue(curr: YMapEntry): Boolean = isNullValue(curr.value)
  private def isNullValue(node: YNode): Boolean     = node.tagType == YType.Null

  private def deepFilterOutNulls(entries: IndexedSeq[YMapEntry], entry: YMapEntry): IndexedSeq[YMapEntry] = {
    if (isNullValue(entry)) entries
    else entries :+ deepFilterOutNulls(entry)
  }

  private def copy(entry: YMapEntry, value: YNode) = YMapEntry(entry.location, IndexedSeq(entry.key, value))

  private def deepFilterOutNulls(entry: YMapEntry): YMapEntry = entry.value.value match {
    case map: YMap =>
      val empty    = IndexedSeq.empty[YMapEntry]
      val filtered = map.entries.foldLeft(empty) { (acc, curr) => deepFilterOutNulls(acc, curr) }
      YMapEntry(entry.location, IndexedSeq(entry.key, YNode(YMap(map.location, filtered))))
    case _ => entry
  }
}
