package org.mulesoft.apb.client.scala

import amf.core.client.common.validation.{ProfileNames, StrictValidationMode}
import amf.core.client.scala.model.document.BaseUnit
import amf.core.client.scala.model.domain.{AmfArray, Shape}
import amf.core.client.scala.parse.document.{ParsedDocument, ParserContext, SyamlParsedDocument}
import amf.core.client.scala.resource.ResourceLoader
import amf.core.client.scala.validation.AMFValidationReport
import amf.core.internal.adoption.IdAdopter
import amf.core.internal.parser.YMapOps
import amf.core.internal.plugins.syntax.SyamlSyntaxParsePlugin
import amf.core.internal.remote.Mimes
import amf.core.internal.unsafe.PlatformSecrets
import amf.shapes.client.scala.JsonLDSchemaElementClient
import amf.shapes.client.scala.config.{JsonLDSchemaConfiguration, JsonSchemaConfiguration}
import amf.shapes.client.scala.model.document.{JsonLDInstanceDocument, JsonSchemaDocument}
import amf.shapes.client.scala.model.domain.jsonldinstance.JsonLDObject
import amf.shapes.internal.document.metamodel.JsonLDInstanceDocumentModel.Encodes
import org.mulesoft.apb.internal.gcl.GclDefaults
import org.mulesoft.apb.internal.loaders.NestedDocumentRL
import org.mulesoft.apb.internal.render.APIInstanceRenderer
import org.mulesoft.apb.project.client.scala.instances.APIInstanceBuilder
import org.mulesoft.apb.project.client.scala.model.DynamicObject.toJsonLDObject
import org.mulesoft.apb.project.client.scala.model.management.APIInstance
import org.mulesoft.apb.project.internal.BuildResult
import org.mulesoft.apb.project.internal.instances.ErrorAPIInstance
import org.mulesoft.apb.project.internal.instances.ResourceKind.Other
import org.mulesoft.common.collections.Group
import org.yaml.model.{YDocument, YMap, YNode}
import org.yaml.parser.YamlParser
import org.yaml.render.JsonRender

import scala.concurrent.ExecutionContext.Implicits.global
import scala.concurrent.Future

class APIInstanceClient private (instance: String, amfConfig: JsonLDSchemaConfiguration) extends PlatformSecrets {
  def build(): Future[BuildResult[BaseUnit]] =
    parseGcls().map(resources => BuildResult(assembleInstances(resources), mergeReports(resources)))

  def render(instance: APIInstance): String = APIInstanceRenderer(instance).render()

  private def parseGcls(): Future[List[BuildResult[JsonLDInstanceDocument]]] = {
    for {
      gclContent <- platform.fetchContent(instance, amfConfig)
      instances  <- parseDocuments(gclContent.stream.toString)
    } yield instances
  }

  private def parseDocuments(gclContent: String): Future[List[BuildResult[JsonLDInstanceDocument]]] = {
    val docs    = parseDocumentsIn(gclContent)
    val grouped = groupGclByKind(docs)
    val instances = grouped.map { case (kind, docs) =>
      parseGclResources(kind, docs.toList)
    }
    Future.sequence(instances).map(_.flatten.toList)
  }

  private def parseGclResources(kind: String, docs: List[YDocument]) = {
    for {
      schema    <- amfConfig.baseUnitClient().parseJsonLDSchema(kind).map(_.jsonDocument)
      instances <- parseInstances(docs, schema)
    } yield {
      instances
    }
  }

  private def groupGclByKind(docs: List[YDocument]) = {
    Group(docs).legacyGroupBy(doc => GclResource(doc.node).kind.getOrElse(Other.kind))
  }

  private def parseDocumentsIn(gclContent: String): List[YDocument] = {
    YamlParser(gclContent).documents().toList
  }

  private def parseInstances(docs: List[YDocument], schema: JsonSchemaDocument) = {
    val docIndex = buildGclUriIndex(docs)
    val client   = buildClient(docIndex)
    val results = docIndex.map { case (uri, document) =>
      for {
        instance <- client.parseJsonLDInstance(uri, schema).map(_.instance)
        adopted = {
          new IdAdopter(instance, this.instance).adoptFromRoot()
          instance
        }
      } yield {
        val report = validate(document, schema.encodes)
        BuildResult(adopted, report)
      }
    }
    Future.sequence(results).map(_.toList)
  }

  private def buildGclUriIndex(docs: List[YDocument]) = {
    docs.zipWithIndex.map { case (doc, i) => s"file://gcl-instance/$i.yaml" -> doc }.toMap
  }

  private def buildClient(docs: Map[String, YDocument]) = {
    val syntaxPlugin = GCLHackedSyntaxPlugin(docs)
    amfConfig
      .withPlugin(syntaxPlugin)
      .withResourceLoader(NestedDocumentRL)
      .baseUnitClient()
  }

  private def validate(doc: YDocument, schema: Shape): AMFValidationReport = {
    val json = JsonRender.render(doc)
    validate(schema, json)
  }

  private def validate(schema: Shape, payload: String): AMFValidationReport = {
    val client = JsonSchemaConfiguration.JsonSchema().elementClient()
    client
      .payloadValidatorFor(schema, Mimes.`application/json`, StrictValidationMode)
      .syncValidate(payload)
  }

  private def mergeReports(resources: List[BuildResult[JsonLDInstanceDocument]]) = {
    resources.map(_.report).foldLeft(AMFValidationReport.empty("", ProfileNames.AMF)) { (acc, curr) => acc.merge(curr) }
  }

  private def assembleInstances(result: List[BuildResult[JsonLDInstanceDocument]]) = {
    val instances = new APIInstanceBuilder(result.map(_.result)).build()
    if (instances.isEmpty) wrap(List(ErrorAPIInstance))
    else wrap(instances.map(toJsonLDObject(_)))
  }

  private def wrap(nodes: Seq[JsonLDObject]) = JsonLDInstanceDocument().setWithoutId(Encodes, AmfArray(nodes))

  case class GclResource(node: YNode) {

    lazy val map: YMap = node.as[YMap]

    def kind: Option[String] = map.key("kind").map(_.value.as[String])
  }

  private case class GCLHackedSyntaxPlugin(index: Map[String, YDocument]) extends SyamlSyntaxParsePlugin {

    override def parse(text: CharSequence, mediaType: String, ctx: ParserContext): ParsedDocument = SyamlParsedDocument(
        index(text.toString)
    )
  }
}

private[apb] object APIInstanceClient {

  def build(instance: String, rl: ResourceLoader): Future[BuildResult[BaseUnit]] =
    apply(instance, None, Some(rl)).build()

  def build(instance: String): Future[BuildResult[BaseUnit]] = apply(instance, None, None).build()

  def render(instance: APIInstance): String = emptyClient.render(instance)

  private def emptyClient: APIInstanceClient = apply("", None, None)

  private[apb] def apply(instance: String, definedBy: Option[String], rl: Option[ResourceLoader]): APIInstanceClient = {
    val amfConfig = computeApiInstanceParameters(definedBy, rl)
    new APIInstanceClient(instance, amfConfig)
  }

  private def computeApiInstanceParameters(definedBy: Option[String], rl: Option[ResourceLoader]) = {
    val baseConfig = jsonLdSchemaConfig(rl)
    definedBy match {
      case Some(_) => baseConfig
      case _       => GclDefaults(baseConfig)
    }
  }

  private def jsonLdSchemaConfig(rl: Option[ResourceLoader]) = {
    val base = JsonLDSchemaConfiguration.JsonLDSchema()
    rl.fold(base)(r => base.withResourceLoader(r))
  }
}
