package org.mulesoft.apb.client.scala.check

import amf.core.client.common.remote.Content
import amf.core.client.common.validation.{ProfileName, SeverityLevels}
import amf.core.client.scala.resource.ResourceLoader
import amf.core.client.scala.validation.{AMFValidationReport, AMFValidationResult}
import org.mulesoft.apb.project.client.scala.descriptor.DescriptorHandler
import org.mulesoft.apb.project.client.scala.model.descriptor.ProjectDescriptor
import org.mulesoft.apb.project.client.scala.model.report.{APBReport, APBResult, ProjectAspects}
import org.mulesoft.apb.project.internal.parser.APBEnv

import scala.concurrent.{ExecutionContext, Future}
import scala.language.implicitConversions

case class APIProjectDirectoryCheck(resourceLoader: ResourceLoader) {
  implicit val executionContext: ExecutionContext = ExecutionContext.Implicits.global

  private def wrapResults(result: Seq[AMFValidationResult]): AMFValidationReport =
    AMFValidationReport("", ProfileName("API Project Check"), result)

  def check(): Future[AMFValidationReport] = {
    fetchFile(APBEnv.descriptorFileName, missingDescriptor)
      .flatMap(processFetched)
      .map(wrapResults)
  }

  private def processFetched(fetched: Either[AMFValidationResult, Content]) = {
    fetched match {
      case Left(r)  => Future.successful(Seq(r))
      case Right(c) => processDescriptor(c.toString())
    }
  }

  private def fetchFile(
      file: String,
      resultBuilder: (String, Throwable) => APBResult
  ): Future[Either[APBResult, Content]] = resourceLoader
    .fetch(file)
    .map(Right(_))
    .recoverWith({ case e: Throwable =>
      Future.successful(Left(resultBuilder(file, e)))
    })

  private def processDescriptor(content: String): Future[Seq[APBResult]] = {
    val result = DescriptorHandler().parse(content)
    checkProjectFiles(result.descriptor).map(filesResults => result.results ++ filesResults)
  }

  private def checkProjectFiles(descriptor: ProjectDescriptor): Future[Seq[APBResult]] = {

    val mainResult           = descriptor.main.map(checkFile(_, missingMain))
    val instancesResults     = checkFiles(descriptor.instances.map(_.gcl), missingGCL)
    val documentationResults = checkFiles(descriptor.documentation.map(_.path), missingDocumentation)
    Future.sequence(mainResult.toSeq ++ instancesResults ++ documentationResults).map(_.flatten)
  }

  private def checkFiles(files: Seq[String], resultBuilder: (String, Throwable) => APBResult) =
    files.map(checkFile(_, resultBuilder))

  private def checkFile(file: String, resultBuilder: (String, Throwable) => APBResult) = {
    fetchFile(file, resultBuilder).map {
      case Left(r) => Some(r)
      case _       => None
    }
  }

  private def missingMain(main: String, e: Throwable): APBResult =
    missingFileResult(s"Missing main file '$main' file at directory", "missing-main-file", e)

  private def missingGCL(gcl: String, e: Throwable): APBResult =
    missingFileResult(s"Missing instance file '$gcl' file at directory", "missing-instance-file", e)

  private def missingDocumentation(path: String, e: Throwable): APBResult =
    missingFileResult(s"Missing documentation file '$path' file at directory", "missing-documentation-file", e)

  private def missingFileResult(message: String, validationId: String, e: Throwable): APBResult =
    APBResult(
        message + s". Message ${e.getMessage}",
        SeverityLevels.VIOLATION,
        "",
        None,
        validationId,
        None,
        None,
        null,
        ProjectAspects.GLOBAL
    )

  private def missingDescriptor(main: String, e: Throwable): APBResult =
    missingFileResult(s"Error fetching main file '$main' file at directory", "missing-main-file", e: Throwable)

}
