package org.mulesoft.apb.internal.client.project

import amf.aml.client.scala.AMLConfiguration
import amf.apicontract.client.scala.APIConfiguration
import amf.core.client.scala.AMFResult
import amf.core.client.scala.config.RenderOptions
import amf.core.client.scala.model.document.Document
import amf.core.client.scala.resource.ResourceLoader
import amf.core.client.scala.validation.AMFValidationReport
import amf.core.internal.remote.Mimes.{`application/json`, `application/ld+json`, `application/yaml`}
import amf.core.internal.unsafe.PlatformSecrets
import amf.shapes.client.scala.model.domain.jsonldinstance.JsonLDObject
import org.mulesoft.apb.client.scala.{APBOptions, APIProjectClient}
import org.mulesoft.apb.internal.client.contract.{APIContractClient, APIContractClientBuilder}
import org.mulesoft.apb.internal.client.instances.APIInstanceClient
import org.mulesoft.apb.internal.convert.ElementConverters.AmfObjectConverter
import org.mulesoft.apb.internal.gcl.SchemaProvider
import org.mulesoft.apb.internal.lint.APIProjectLinter
import org.mulesoft.apb.project.client.scala.dependency.UnitCacheBuilder
import org.mulesoft.apb.project.client.scala.descriptor.DescriptorParseResult
import org.mulesoft.apb.project.client.scala.environment.DependencyFetcher
import org.mulesoft.apb.project.client.scala.extensions.APIProjectExtension
import org.mulesoft.apb.project.client.scala.model.descriptor.documentation.Documentation
import org.mulesoft.apb.project.client.scala.model.descriptor.community.Community
import org.mulesoft.apb.project.client.scala.model.descriptor.{Gav, Instance, ProjectDescriptor}
import org.mulesoft.apb.project.client.scala.model.project.Project
import org.mulesoft.apb.project.client.scala.model.report.{APBReport, APBResult}
import org.mulesoft.apb.project.client.scala.model.{BaseUnitBuildResult, ProjectBuildResult}
import org.mulesoft.apb.project.client.scala.{DependencySet, DependencySetResult}
import org.mulesoft.apb.project.internal.dependency.DependencySetParser
import org.mulesoft.apb.project.internal.descriptor.ApiProjectNamespaces.aliases
import org.mulesoft.apb.project.internal.idadoption.APBIdAdopter
import org.mulesoft.apb.project.internal.idadoption.URITools.URIStr
import org.mulesoft.apb.project.internal.instances.{ExtensionAssetParser, ScopedExtensionIndex}
import org.mulesoft.apb.project.internal.model.project.ProjectDocumentBuilder
import org.mulesoft.apb.project.internal.render.plugin.ProjectSummaryRenderPlugin
import org.mulesoft.apb.project.internal.view.ApiSummaryView

import scala.concurrent.ExecutionContext.Implicits.global
import scala.concurrent.Future
import scala.util.Success

// TODO native-jsonld: change policies for a list of generic runtime dependencies with their own definition for merging
class DefaultAPIProjectClient private[apb] (
    descriptorFinder: DescriptorFinder,
    dependencyFetcher: DependencyFetcher,
    resourceLoaders: List[ResourceLoader],
    unitCacheBuilder: UnitCacheBuilder,
    extensions: Seq[APIProjectExtension],
    base: Option[String] = None,
    options: APBOptions = APBOptions()
) extends APIProjectClient
    with PlatformSecrets {

  private var _project: Option[ProjectBuildResult]       = None
  private var _dependencies: Option[DependencySetResult] = None
  private var _descriptor: Option[DescriptorParseResult] = None

// TODO: enable when descriptor date formats are fixed.

//  override def descriptor(): Future[DescriptorParseResult] = whenInvalid(throwInvalidDescriptorError) { () =>
//    memoizedDescriptor { () =>
//      descriptorFinder.find()
//    }
//  }

  override def descriptor(): Future[DescriptorParseResult] = memoizedDescriptor { () =>
    descriptorFinder.find()
  }

  override def dependencies(): Future[DependencySetResult] = memoizedDependencies { () =>
    for {
      descriptor   <- descriptor()
      dependencies <- new DependencySetParser(dependencyFetcher, resourceLoaders).build(descriptor)
    } yield {
      dependencies
    }
  }

  def project(): Future[ProjectBuildResult] = memoizeProject { () =>
    for {
      descriptor <- descriptor()
      deps       <- dependencies()
      result     <- project(descriptor.descriptor, deps)
    } yield {
      result
    }
  }

  private def project(
      descriptor: ProjectDescriptor,
      dependenciesResult: DependencySetResult
  ): Future[ProjectBuildResult] =
    memoizeProject { () =>
      val builder = ProjectDocumentBuilder(dependenciesResult).withExtensions(extensions)
      for {
        contractOption <- buildContract(dependenciesResult)
        builtInstances <- buildInstances(descriptor)
      } yield {
        val (instances, instancesError) = extractJsonLDNodes(builtInstances)
        contractOption.foreach(co => builder.withContract(co.result, co.results))
        builder.withInstances(instances, instancesError)

        val documentations = buildDocumentation(descriptor)
        builder.withDocumentation(documentations, Nil)

        val communities = buildCommunities(descriptor)
        builder.withCommunities(communities, Nil)

        base.foreach(builder.withBase)
        builder.build()
      }
    }

  def validate(): Future[APBReport] = {
    for {
      deps             <- dependencies()
      model            <- project()
      validationReport <- validate(model.project, deps.dependencySet)
    } yield new APBReport(model.results ++ validationReport)
  }

  private def validate(project: Project, dependencies: DependencySet): Future[Seq[APBResult]] = {
    dependencies
      .descriptor()
      .main
      .map(new APIContractClient(dependencies, resourceLoaders, unitCacheBuilder, _))
      .map(_.validate(project.apiContract()).map(_.results.map(APBResult.forContract)))
      .getOrElse(Future.successful(Seq.empty))
    // instance validation?
  }

  def serialize(): Future[String] = {
    for {
      model <- project()
    } yield {
      buildConfig.baseUnitClient().render(model.project.document)
    }
  }

  private def serialize(project: Project): String = buildConfig.baseUnitClient().render(project.document)

  def summary(schemaBase: String, format: String = `application/ld+json`): Future[String] = for {
    model <- project()
  } yield summaryFromProject(schemaBase, model.project, format)

  private def summaryFromProject(schemaBase: String, project: Project, format: String = `application/ld+json`): String =
    format match {
      case `application/ld+json` => summaryInJsonLd(schemaBase, project)
      case `application/json`    => summaryInYamlLike(project, format)
      case `application/yaml`    => summaryInYamlLike(project, format)
      case _                     => throw new IllegalArgumentException("format: $format is not supported")
    }

  private def summaryInJsonLd(schemaBase: String, project: Project): String = {
    val summary = ApiSummaryView(schemaBase)
    summary.view(project.projectInfo)
  }

  private def summaryInYamlLike(project: Project, format: String): String = {
    AMLConfiguration
      .predefined()
      .withPlugins(List(ProjectSummaryRenderPlugin))
      .baseUnitClient()
      .render(project.document, format)
  }

  def lint(): Future[List[AMFValidationReport]] = {
    for {
      deps    <- dependencies()
      model   <- serialize()
      reports <- projectLinter(model).lintProfiles(deps.dependencySet.validation().map(_.profile).toList)
    } yield {
      reports
    }
  }

  def lint(rulesets: List[Gav]): Future[List[AMFValidationReport]] = {
    for {
      descriptorResult <- descriptor()
      deps             <- dependencies()
      project          <- project(descriptorResult.descriptor, deps)
      reports          <- lint(rulesets, project.project)
    } yield {
      reports
    }
  }

  private def lint(rulesets: List[Gav], project: Project): Future[List[AMFValidationReport]] = {
    val graph = serialize(project)
    projectLinter(graph).lint(rulesets)
  }

  private def projectLinter(graph: String) = new APIProjectLinter(dependencyFetcher, graph)

  private def extractJsonLDNodes(units: Seq[BaseUnitBuildResult]) =
    (units.flatMap(_.result.toJsonLDObjects()).toList, units.flatMap(_.results))

  private def buildContractFromDescriptor(
      dependenciesResult: DependencySetResult
  ): Future[Option[BaseUnitBuildResult]] = {
    APIContractClientBuilder(dependencyFetcher)
      .withResourceLoaders(resourceLoaders)
      .build(dependenciesResult)
      .build(withIdShortening = false)
      .map { result =>
        if (options.enforceMainAPIContract) {
          createValidContractResult(result)
        } else Some(BaseUnitBuildResult(result))
      }
  }

  /** Checks that the [[AMFResult]] contains a valid base unit for contracts. Only [[Document]] can be contracts.
    *
    * @param initialResult
    *   raw [[AMFResult]] that results from building the contract found in 'main' file.
    * @return
    */
  private def createValidContractResult(initialResult: AMFResult): Option[BaseUnitBuildResult] = initialResult match {
    case result @ AMFResult(_: Document, _) => Some(BaseUnitBuildResult(result))
    case _                                  => None
  }

  private def buildInstance(instance: Instance, index: ScopedExtensionIndex): Future[BaseUnitBuildResult] = {
    val client = APIInstanceClient(instance, resourceLoaders, index)
    client.build()
  }

  private def buildInstances(descriptor: ProjectDescriptor): Future[Seq[BaseUnitBuildResult]] = {
    for {
      extensionSchema <- SchemaProvider.allExtensionSchemas
      extensionIndex <- ExtensionAssetParser(dependencyFetcher, extensionSchema.toMap)
        .parse(
          descriptor
        ) // TODO: move parsing to project configuration building (are fixed, cannot change)
        .map(_.scoped(descriptor.dependencies().map(_.gav).toSet))
      instances <- Future.sequence(
        descriptor.instances.map(instance => buildInstance(instance, extensionIndex))
      )
    } yield {
      instances
    }
  }

  private def buildContract(dependenciesResult: DependencySetResult): Future[Option[BaseUnitBuildResult]] = {
    dependenciesResult.dependencySet.descriptor().main match {
      case Some(_) =>
        buildContractFromDescriptor(dependenciesResult)
      case _ => Future.successful(None)
    }
  }

  private def buildDocumentation(descriptor: ProjectDescriptor): List[JsonLDObject] =
    descriptor.documentation.map(doc => buildDocumentationNode(doc)).toList

  private def buildDocumentationNode(docNode: Documentation): JsonLDObject = {
    new APBIdAdopter(docNode.path.fromPath).adoptFromRelative(docNode.internal)
    docNode.internal
  }

  private def buildCommunities(descriptor: ProjectDescriptor): List[JsonLDObject] =
    descriptor.communities.map(doc => buildCommunity(doc)).toList

  private def buildCommunity(docNode: Community): JsonLDObject = {
    // ToDo: adopt ids correctly
    new APBIdAdopter(docNode.portal + docNode.version.fold("")(version => s"/$version"))
      .adoptFromRelative(docNode.internal)
    docNode.internal
  }

  private def buildConfig = {
    APIConfiguration
      .API()
      .withRenderOptions(RenderOptions().withPrettyPrint.withCompactUris.withGovernanceMode)
      .withAliases(aliases)
  }

  private def memoizeProject(provider: () => Future[ProjectBuildResult]): Future[ProjectBuildResult] = {
    memoized(_project, provider, cacheProject)
  }

  private def memoizedDependencies(provider: () => Future[DependencySetResult]): Future[DependencySetResult] = {
    memoized(_dependencies, provider, cacheDependencies)
  }

  private def memoizedDescriptor(provider: () => Future[DescriptorParseResult]): Future[DescriptorParseResult] = {
    memoized(_descriptor, provider, cacheDescriptor)
  }

  private def memoized[T](memoizedValue: Option[T], provider: () => Future[T], memoizer: T => Unit): Future[T] = {
    memoizedValue match {
      case Some(project) => Future.successful(project)
      case None =>
        provider().andThen { case Success(project) =>
          memoizer(project)
        }
    }
  }

  private def cacheProject(result: ProjectBuildResult): Unit = {
    if (result.conforms) {
      this._project = Some(result)
    }
  }

  private def cacheDescriptor(result: DescriptorParseResult): Unit = {
    if (result.conforms) {
      this._descriptor = Some(result)
    }
  }

  private def cacheDependencies(dependenciesResult: DependencySetResult): Unit = {
    this._dependencies = Some(
      dependenciesResult
    ) // TODO: question, do we always cache although we have errors in dependency fetching ???
  }
}
