package org.mulesoft.apb.internal.gcl

import amf.core.client.common.transform.PipelineId
import amf.core.client.scala.AMFResult
import amf.shapes.client.scala.config.JsonLDSchemaConfiguration
import amf.shapes.client.scala.model.document.JsonSchemaDocument
import org.mulesoft.apb.internal.generated._
import org.mulesoft.apb.project.internal.instances.ResourceKind._
import org.mulesoft.apb.project.internal.instances.ResourceVersion
import org.mulesoft.apb.project.internal.instances.ResourceVersion._

import scala.concurrent.ExecutionContext.Implicits.global
import scala.concurrent.Future

object SchemaProvider {

  def get(kind: String): Option[String] = if (kind == "common.json") Some(commonSchema) else get(kind, default.version)

  def get(kind: String, version: String): Option[String] = schemas.get(version + kind)

  def all: Map[String, String] = schemas

  def extensions(): Map[String, String] = extensionSchemas

  lazy val schemasToVersionMap: Map[String, Seq[String]] =
    (gatewayV1Alpha1Schemas.keys.map(_ -> gatewayV1Alpha1.version) ++
      gatewayV1Beta1Schemas.keys.map(_ -> gatewayV1Beta1.version) ++
      monitoringV1Alpha1Schemas.keys.map(_ -> monitoringV1Alpha1.version) ++
      networkingV1Alpha1Schemas.keys.map(_ -> networkingV1Alpha1.version) ++
      defaultSchemas.keys.map(_ -> default.version))
      .groupBy(_._1)
      .mapValues(t => t.map(_._2).toSeq)

  lazy val allExtensionSchemas: Future[Seq[(String, JsonSchemaDocument)]] =
    Future.sequence(
      Seq(
        gatewayV1Alpha1Extension.map(gatewayV1Alpha1.version -> _),
        gatewayV1Beta1Extension.map(gatewayV1Beta1.version -> _),
        networkV1Alpha1Extension.map(networkingV1Alpha1.version -> _),
        defaultExtension.map(default.version -> _)
      )
    )

  private lazy val gatewayV1Alpha1Extension: Future[JsonSchemaDocument] = getExtensionSchema(gatewayV1Alpha1.version)
  private lazy val gatewayV1Beta1Extension: Future[JsonSchemaDocument]  = getExtensionSchema(gatewayV1Beta1.version)
  private lazy val networkV1Alpha1Extension: Future[JsonSchemaDocument] = getExtensionSchema(networkingV1Alpha1.version)
  private lazy val defaultExtension: Future[JsonSchemaDocument]         = getExtensionSchema(default.version)

  private val gatewayV1Alpha1Extensions: Map[String, String] = Map(
    Extension.kind -> ExtensionSchema.gatewayV1Alpha1
  )

  private val gatewayV1Beta1Extensions: Map[String, String] = Map(
    Extension.kind -> ExtensionSchema.gatewayV1Beta1
  )

  private val networkV1Alpha1Extensions: Map[String, String] = Map(
    Extension.kind -> ExtensionSchema.networkingV1Alpha1
  )

  private val defaultExtensions: Map[String, String] = Map(
    Extension.kind -> ExtensionSchema.default
  )

  private val extensionSchemas: Map[String, String] = mapWithVersion(gatewayV1Alpha1Extensions, gatewayV1Alpha1) ++
    mapWithVersion(gatewayV1Beta1Extensions, gatewayV1Beta1) ++
    mapWithVersion(networkV1Alpha1Extensions, networkingV1Alpha1) ++
    mapWithVersion(defaultExtensions, default)

  private val gatewayV1Alpha1Schemas: Map[String, String] = Map(
    ApiInstance.kind        -> ApiInstanceSchema.gatewayV1Alpha1,
    EnvironmentBinding.kind -> EnvironmentBindingSchema.gatewayV1Alpha1,
    Environment.kind        -> EnvironmentSchema.gatewayV1Alpha1,
    Extension.kind          -> ExtensionSchema.gatewayV1Alpha1,
    PolicyBinding.kind      -> PolicyBindingSchema.gatewayV1Alpha1,
    Service.kind            -> ServiceSchema.gatewayV1Alpha1
  )

  private val commonSchema = Common.schema

  private val gatewayV1Beta1Schemas: Map[String, String] = Map(
    Extension.kind -> ExtensionSchema.gatewayV1Beta1
  )

  private val monitoringV1Alpha1Schemas: Map[String, String] = Map(
    Alert.kind -> AlertSchema.monitoringV1Alpha1
  )

  private val networkingV1Alpha1Schemas: Map[String, String] = Map(
    ApiInstance.kind   -> ApiInstanceSchema.networkingV1Alpha1,
    Extension.kind     -> ExtensionSchema.networkingV1Alpha1,
    PolicyBinding.kind -> PolicyBindingSchema.networkingV1Alpha1
  )

  private val defaultSchemas: Map[String, String] = Map(
    ApiInstance.kind        -> ApiInstanceSchema.default,
    EnvironmentBinding.kind -> EnvironmentBindingSchema.default,
    Environment.kind        -> EnvironmentSchema.default,
    Extension.kind          -> ExtensionSchema.default,
    PolicyBinding.kind      -> PolicyBindingSchema.default,
    Service.kind            -> ServiceSchema.default,
    Alert.kind              -> AlertSchema.default
  )

  private val schemas: Map[String, String] =
    mapWithVersion(gatewayV1Alpha1Schemas, gatewayV1Alpha1) ++
      mapWithVersion(gatewayV1Beta1Schemas, gatewayV1Beta1) ++
      mapWithVersion(monitoringV1Alpha1Schemas, monitoringV1Alpha1) ++
      mapWithVersion(networkingV1Alpha1Schemas, networkingV1Alpha1) ++
      mapWithVersion(defaultSchemas, default)

  private def getExtensionSchema(version: String): Future[JsonSchemaDocument] = {
    val configuration = (GCLSchemaDefaultRL.extensionLoaders ++ GCLSchemaDefaultRL.commonLoader).foldLeft(
      JsonLDSchemaConfiguration.JsonLDSchema()
    )((config, rl) => config.withResourceLoader(rl))
    configuration
      .baseUnitClient()
      .parse(version + Extension.kind)
      .map(transform(_, configuration))
      .map(_.baseUnit.asInstanceOf[JsonSchemaDocument])
  }

  private def transform(parsed: AMFResult, configuration: JsonLDSchemaConfiguration): AMFResult = {
    val transformed = configuration.baseUnitClient().transform(parsed.baseUnit, PipelineId.Editing)
    transformed.copy(results = parsed.results ++ transformed.results)
  }

  private def mapWithVersion(schemasKindMap: Map[String, String], version: ResourceVersion): Map[String, String] =
    schemasKindMap.map(s => (version.version + s._1, s._2))

}
