package org.mulesoft.apb.client.scala.instances

import amf.core.client.common.validation.{SeverityLevels, StrictValidationMode}
import amf.core.client.scala.model.domain.{AmfArray, Shape}
import amf.core.client.scala.parse.document.{ParsedDocument, ParserContext, SyamlParsedDocument}
import amf.core.client.scala.resource.ResourceLoader
import amf.core.internal.parser.YMapOps
import amf.core.internal.plugins.syntax.SyamlSyntaxParsePlugin
import amf.core.internal.remote.Mimes
import amf.core.internal.unsafe.PlatformSecrets
import amf.shapes.client.scala.config.{JsonLDSchemaConfiguration, JsonSchemaConfiguration}
import amf.shapes.client.scala.model.document.{JsonLDInstanceDocument, JsonSchemaDocument}
import amf.shapes.client.scala.model.domain.jsonldinstance.JsonLDObject
import amf.shapes.internal.document.metamodel.JsonLDInstanceDocumentModel.Encodes
import org.mulesoft.apb.internal.gcl.SchemaProvider.sanitizeSchemaNames
import org.mulesoft.apb.internal.gcl.{GclDefaults, SchemaProvider}
import org.mulesoft.apb.internal.loaders.NestedDocumentRL
import org.mulesoft.apb.project.client.scala.model.descriptor.Instance
import org.mulesoft.apb.project.client.scala.model.report.{APBResult, ProjectAspects}
import org.mulesoft.apb.project.client.scala.model.{BaseUnitBuildResult, JsonLDInstanceDocumentBuildResult}
import org.mulesoft.apb.project.internal.idadoption.IdAdopterProvider
import org.mulesoft.apb.project.internal.idadoption.URITools._
import org.mulesoft.apb.project.internal.instances.ResourceKind.Other
import org.mulesoft.apb.project.internal.instances._
import org.mulesoft.apb.project.internal.model.GraphAccessors.toJsonLDObject
import org.mulesoft.apb.project.internal.validations.ProjectValidations
import org.mulesoft.common.collections.Group
import org.yaml.model.{YDocument, YMap, YNode}
import org.yaml.parser.YamlParser
import org.yaml.render.{JsonRender, JsonRenderOptions}

import scala.collection.mutable.ArrayBuffer
import scala.concurrent.ExecutionContext.Implicits.global
import scala.concurrent.Future

class APIInstanceClient private (
    instance: Instance,
    amfConfig: JsonLDSchemaConfiguration,
    extensionIndex: ScopedExtensionIndex
) extends PlatformSecrets {
  def build(): Future[BaseUnitBuildResult] = {
    parseGcls().map(resources => buildResultFromResources(resources))
  }
  private def buildResultFromResources(resources: List[JsonLDInstanceDocumentBuildResult]): BaseUnitBuildResult = {
    BaseUnitBuildResult(assembleInstances(resources, extensionIndex), resources.flatMap(_.results))
  }

  private def parseGcls(): Future[List[JsonLDInstanceDocumentBuildResult]] = {
    for {
      gclContent <- platform.fetchContent(instance.gcl, amfConfig)
      instances  <- parseDocuments(gclContent.stream.toString)
    } yield instances
  }

  private[apb] def parseDocuments(gclContent: String): Future[List[JsonLDInstanceDocumentBuildResult]] = {
    val docs    = parseDocumentsIn(gclContent)
    val grouped = groupGclByKindAndVersion(docs)
    val instances = grouped.map { case ((version, kind), docs) =>
      checkAndParseDocsByResource(version, kind, docs)
    }
    Future.sequence(instances).map(_.flatten.toList)
  }

  private def checkAndParseDocsByResource(
      version: String,
      kind: String,
      docs: ArrayBuffer[YDocument]
  ): Future[List[JsonLDInstanceDocumentBuildResult]] = {
    if (!isValidKind(kind)) {
      Future.successful(buildInvalidResult("kind", kind, Nil, docs).toList)
    } else {
      val sanitizedVersion = sanitizeVersion(version)
      validateKindVersionSet(kind, sanitizedVersion) match {
        case Right(_)       => parseGclResources(sanitizedVersion, kind, docs.toList)
        case Left(versions) => Future.successful(buildInvalidResult("version", version, versions, docs).toList)
      }
    }
  }

  private def sanitizeVersion(version: String): String = if (version == "v1") version.toUpperCase else version

  private def isValidKind(kind: String) = ResourceKind.ValidKinds.exists(_.kind == kind)

  private def validateKindVersionSet(kind: String, version: String): Either[Seq[String], Unit] = {
    val schemas = SchemaProvider.schemasToVersionMap
    schemas
      .get(kind)
      .map { versions =>
        if (versions.contains(version)) {
          Right()
        } else {
          // I accept V1 as a WA, I don't want it listed as valid
          Left(versions.filterNot(_ == "V1"))
        }
      }
      .getOrElse(Left(Nil))
  }

  private def buildInvalidResult(
      field: String,
      value: String,
      validValues: Seq[String] = Nil,
      docs: ArrayBuffer[YDocument]
  ) = {
    docs.map { _ =>
      val errorDocument = JsonLDInstanceDocument().withId(instance.gcl.withProtocol)
      JsonLDInstanceDocumentBuildResult(
        errorDocument,
        Seq(buildErrorFor(errorDocument, field, value, validValues))
      )
    }
  }

  private def buildErrorFor(node: JsonLDInstanceDocument, field: String, value: String, validValues: Seq[String]) = {
    val baseMessage = s"Invalid $field: $value"
    val message =
      if (validValues.isEmpty) baseMessage else baseMessage + s". Valid options are [${validValues.mkString(", ")}]"
    APBResult(
      message,
      SeverityLevels.VIOLATION,
      node.id,
      None,
      ProjectValidations.GCLSyntaxError.id,
      None,
      Some(instance.gcl),
      null,
      ProjectAspects.RUNTIME
    )
  }

  private def parseGclResources(version: String, kind: String, docs: List[YDocument]) = {
    for {
      schema    <- amfConfig.baseUnitClient().parseJsonLDSchema(sanitizeSchemaNames(version + kind)).map(_.jsonDocument)
      instances <- parseInstances(docs, schema, kind)
    } yield {
      instances
    }
  }

  private def groupGclByKindAndVersion(docs: List[YDocument]): Map[(String, String), ArrayBuffer[YDocument]] = {
    Group(docs).legacyGroupBy { doc =>
      val gcl     = GclResource(doc.node)
      val kind    = gcl.kind.getOrElse(Other.kind)
      val version = gcl.version.getOrElse(ResourceVersion.default.version)
      (version, kind)
    }
  }

  private def parseDocumentsIn(gclContent: String): List[YDocument] = {
    YamlParser(gclContent, formattedInstancePath).documents().toList
  }

  private def parseInstances(docs: List[YDocument], schema: JsonSchemaDocument, kind: String) = {
    val docIndex = buildGclUriIndex(docs, kind)
    val client   = buildClient(docIndex)
    val results = docIndex.map { case (_, documentWithUri) =>
      for {
        // we send to AMF the path uri without encoding it, because AMF will always encode it
        instance <- client.parseJsonLDInstance(documentWithUri.unencodedUri, schema).map(_.instance)
      } yield {
        val results = validate(documentWithUri.document, schema.encodes)
        JsonLDInstanceDocumentBuildResult(instance, results)
      }
    }
    Future.sequence(results).map(_.toList)
  }

  private val formattedInstancePath = this.instance.gcl.withProtocol

  private case class DocumentWithUri(unencodedUri: String, document: YDocument)

  private def buildGclUriIndex(docs: List[YDocument], kind: String): Map[String, DocumentWithUri] = {
    toGCLUriList(docs, kind)
      // we encode the path uri to match what AMF will request
      .map(index => (platform.encodeURI(index._1), DocumentWithUri(index._1, index._2)))
      .toMap
  }

  private def toGCLUriList(docs: List[YDocument], kind: String): List[(String, YDocument)] = {
    if (ResourceKind.ApiInstance.kind == kind) docs.map(d => formattedInstancePath -> d)
    else if (docs.size < 2) docs.map(d => formattedInstancePath.forKind(kind) -> d)
    else docs.zipWithIndex.map { case (d, index) => formattedInstancePath.indexed(index, kind) -> d }
  }

  private def buildClient(docs: Map[String, DocumentWithUri]) = {
    val syntaxPlugin = GCLHackedSyntaxPlugin(docs)
    amfConfig
      .withPlugin(syntaxPlugin)
      .withResourceLoader(NestedDocumentRL)
      .withIdAdopterProvider(IdAdopterProvider)
      .baseUnitClient()
  }

  private def validate(doc: YDocument, schema: Shape): Seq[APBResult] = {
    val json = JsonRender.render(doc, 0, JsonRenderOptions().withoutNonAsciiEncode)
    validate(schema, json)
  }

  private def validate(schema: Shape, payload: String): Seq[APBResult] = {
    val client = JsonSchemaConfiguration.JsonSchema().elementClient()
    client
      .payloadValidatorFor(schema, Mimes.`application/json`, StrictValidationMode)
      .syncValidate(payload)
      .results
      .map(_.copy(validationId = ProjectValidations.GCLSyntaxError.id))
      .map(APBResult.forRuntime)
  }

  private def assembleInstances(result: List[JsonLDInstanceDocumentBuildResult], index: ScopedExtensionIndex) = {
    val instances = new APIInstanceBuilder(result.map(_.result), index).build()
    if (instances.isEmpty) wrap(List(ErrorAPIInstance))
    else wrap(instances.map(toJsonLDObject(_)))
  }

  private def wrap(nodes: Seq[JsonLDObject]) = JsonLDInstanceDocument().setWithoutId(Encodes, AmfArray(nodes))

  case class GclResource(node: YNode) {

    lazy val map: YMap = node.as[YMap]

    def kind: Option[String] = map.key("kind").map(_.value.as[String])

    def version: Option[String] = map.key("apiVersion").map(_.value.as[String])
  }

  private case class GCLHackedSyntaxPlugin(index: Map[String, DocumentWithUri]) extends SyamlSyntaxParsePlugin {
    override def parse(text: CharSequence, mediaType: String, ctx: ParserContext): ParsedDocument =
      SyamlParsedDocument(index(text.toString).document)
  }
}

private[apb] object APIInstanceClient extends PlatformSecrets {

  def build(instanceIri: Instance, resourceLoaders: List[ResourceLoader]): Future[BaseUnitBuildResult] =
    apply(instanceIri, resourceLoaders, ScopedExtensionIndex.Empty).build()

  def build(instanceIri: String): Future[BaseUnitBuildResult] = {
    apply(Instance(instanceIri), platform.loaders().toList, ScopedExtensionIndex.Empty).build()
  }

  private[apb] def apply(
      instance: Instance,
      resourceLoaders: List[ResourceLoader],
      index: ScopedExtensionIndex
  ): APIInstanceClient = {
    val amfConfig = computeApiInstanceParameters(instance.definedBy, resourceLoaders)
    new APIInstanceClient(instance, amfConfig, index.addScope(instance.dependencies.map(_.gav).toSet))
  }

  private def computeApiInstanceParameters(definedBy: Option[String], resourceLoaders: List[ResourceLoader]) = {
    val baseConfig = jsonLdSchemaConfig(resourceLoaders)
    definedBy match {
      case Some(_) => baseConfig
      case _       => GclDefaults(baseConfig)
    }
  }

  private def jsonLdSchemaConfig(resourceLoaders: List[ResourceLoader]) = {
    val base = JsonLDSchemaConfiguration.JsonLDSchema()
    base.withResourceLoaders(resourceLoaders)
  }
}
