package org.mulesoft.apb.internal.client.project

import amf.core.client.common.validation.ValidationMode
import amf.core.client.scala.model.domain.Shape
import amf.core.internal.remote.Mimes
import amf.shapes.client.scala.ShapesConfiguration
import amf.shapes.client.scala.model.document.JsonSchemaDocument
import org.mulesoft.apb.project.client.scala.DependencySet
import org.mulesoft.apb.project.client.scala.dependency.ManagementDependency
import org.mulesoft.apb.project.client.scala.helper.GraphHelper
import org.mulesoft.apb.project.client.scala.model.project.Project
import org.mulesoft.apb.project.client.scala.model.report.APBResult
import org.mulesoft.apb.project.internal.validations.ProjectValidations

object ProjectValidatorHelper {

  def validatePolicyDefinitions(project: Project, dependencies: DependencySet): Seq[APBResult] = {
    val policyDefinitions = dependencies.management() // currently is the only kind of management dependencies supported
    val results = project
      .instances()
      .flatMap(
        _.policyBindings.flatMap(pb => {
          val maybePolicyRef    = pb.getPolicyRefName
          val maybePolicyConfig = pb.getConfigOption
          (maybePolicyRef, maybePolicyConfig) match {
            case (Some(policyRef), Some(policyConfig)) => // Validation will performed only if both, ref and config, are defined
              getPolicyDefinition(policyRef, policyDefinitions)
                .map(pd => {
                  val config = GraphHelper.renderToJson(policyConfig)
                  validateSchemaConformance(pd.encodes, config).map(APBResult.forGlobal)
                })
            case _ => Nil // Cannot validate if ref or config are not defined, so it is ignored
          }
        })
      )
      .flatten
    results
  }

  private def getPolicyDefinition(
      ref: String,
      policies: Seq[ManagementDependency]
  ): Option[JsonSchemaDocument] = {
    policies
      .find(p => ref.contains(p.descriptor.gav().assetId) && ref.contains(p.descriptor.gav().version))
      .map(_.baseUnit)
  }

  private def validateSchemaConformance(schema: Shape, content: String) = {
    ShapesConfiguration
      .predefined()
      .elementClient()
      .payloadValidatorFor(schema, Mimes.`application/json`, ValidationMode.StrictValidationMode)
      .syncValidate(content)
      .results
      .distinct
      .map(_.copy(validationId = ProjectValidations.PolicyConfigurationError.id))
  }

}
