package org.mulesoft.apb.internal.client.contract

import amf.apicontract.client.scala.{AMFConfiguration, APIConfiguration, UnrecognizedSpecException}
import amf.core.client.common.transform.PipelineId
import amf.core.client.common.validation.SeverityLevels
import amf.core.client.scala.config.RenderOptions
import amf.core.client.scala.model.document.BaseUnit
import amf.core.client.scala.resource.ResourceLoader
import amf.core.client.scala.validation.{AMFValidationReport, AMFValidationResult}
import amf.core.client.scala.{AMFParseResult, AMFResult}
import amf.core.internal.remote.{Mimes, Spec}
import amf.custom.validation.client.ProfileValidatorNodeBuilder
import amf.custom.validation.internal.report.parser.EmptyProfile
import org.mulesoft.apb.project.client.scala.DependencySet
import org.mulesoft.apb.project.client.scala.config.AMFConfigurationFactory
import org.mulesoft.apb.project.client.scala.config.AMFConfigurationFactory.{createFromClassifier, createSpecificationParsingConfig}
import org.mulesoft.apb.project.client.scala.dependency.UnitCacheBuilder
import org.mulesoft.apb.project.client.scala.model.descriptor.{Gav, ProjectDescriptor}
import org.mulesoft.apb.project.internal.idadoption.URITools._

import scala.concurrent.ExecutionContext.Implicits.global
import scala.concurrent.Future
import scala.language.postfixOps

class APIContractClient private[apb] (
    val dependencySet: DependencySet,
    val resourceLoaders: List[ResourceLoader],
    val unitCacheBuilder: UnitCacheBuilder,
    val mainFile: String
) {

  val descriptor: ProjectDescriptor = dependencySet.descriptor()

  implicit class AMFHelper(config: AMFConfiguration) {
    def parse(): Future[AMFParseResult] = {
      val client     = config.baseUnitClient()
      val uriToParse = mainFile.withProtocol
      client.parse(uriToParse)
    }

    def resolve(bu: BaseUnit, shortenIds: Boolean = true): AMFResult = {
      val pipeline = if (shortenIds) PipelineId.Editing else PipelineId.Cache
      config.baseUnitClient().transform(bu, pipeline)
    }

    def validate(bu: BaseUnit): Future[AMFValidationReport] = config.baseUnitClient().validate(bu)
  }

  implicit class AMFResultHelper(result: AMFResult) {
    def addResults(other: AMFResult): AMFResult = result.copy(results = result.results ++ other.results)
  }

  private def parse(): Future[(AMFParseResult, Spec)] = {
    createSpecificationParsingConfig(dependencySet, resourceLoaders, unitCacheBuilder)
      .parse()
      .map(u => (u, u.sourceSpec))
  }

  private def resolve(r: AMFResult, spec: Spec): AMFResult =
    AMFConfigurationFactory
      .createFromSpec(dependencySet, spec, resourceLoaders, unitCacheBuilder)
      .resolve(r.baseUnit)

  private def validate(r: AMFResult, spec: Spec): Future[AMFValidationReport] =
    validate(r.baseUnit, spec)
  private def validate(baseUnit: BaseUnit, spec: Spec): Future[AMFValidationReport] =
    AMFConfigurationFactory
      .createFromSpec(dependencySet, spec, resourceLoaders, unitCacheBuilder)
      .validate(baseUnit.cloneUnit())

  private def serialize(bu: BaseUnit, withSourceMaps: Boolean = false): String = {
    val options = if (withSourceMaps) RenderOptions().withSourceMaps else RenderOptions().withGovernanceMode
    serialize(bu, options)
  }

  private def serialize(bu: BaseUnit, renderOptions: RenderOptions): String = {
    val configuration = createFromClassifier(descriptor).withRenderOptions(renderOptions)
    AMFConfigurationFactory
      .createFrom(
        dependencySet,
        configuration,
        resourceLoaders,
        unitCacheBuilder
      )
      .baseUnitClient()
      .render(bu, Mimes.`application/ld+json`)
  }

  private def lint(unit: BaseUnit): Future[AMFValidationReport] =
    for {
      reports <- {
        val jsonld = serialize(unit, withSourceMaps = true)
        val eventualReports: Seq[Future[AMFValidationReport]] = dependencySet
          .validation()
          .toList
          .map { profileDep =>
            ProfileValidatorNodeBuilder
              .validator(profileDep.profile)
              .validate(jsonld, descriptor.main.get)
          }
        Future.sequence(eventualReports)
      }
    } yield {
      val empty = AMFValidationReport.empty(unit.id, EmptyProfile)
      reports.foldLeft(empty)((acc, curr) => acc.merge(curr))
    }

  def build(withIdShortening: Boolean = true): Future[AMFResult] = {
    for {
      (result, spec) <- parse()
    } yield {
      try {
        val transformConfig =
          AMFConfigurationFactory.createFromSpec(dependencySet, spec, resourceLoaders, unitCacheBuilder)
        val transformResult = transformConfig.resolve(result.baseUnit, withIdShortening)
        transformResult.copy(results = transformResult.results ++ result.results)
      } catch {
        case _: UnrecognizedSpecException =>
          // Should we reaffirm the parsing result with these assertions?
          //
          // Note: `configFor(spec) will throw UnrecognizedSpecException for specs other than RAML08, RAML10, OAS20, OAS30
          // this includes Async, GraphQL, GRPC

//          assert(result.baseUnit.isInstanceOf[ExternalFragment], "")
//          assert(result.results.length == 1, "")
//          assert(result.results.head.validationId == CouldntGuessRoot.id, "")
          result
        case e: Exception =>
          val error = AMFValidationResult(
            s"Unexpected error processing main file: ${e.getMessage}",
            SeverityLevels.VIOLATION,
            mainFile,
            None,
            "apb-main-parse",
            None,
            Some(mainFile),
            null
          )
          result.copy(results = result.results :+ error)
      }
    }
  }

  def compile(): Future[AMFParseResult] = parse().map(_._1)

  def report(): Future[AMFValidationReport] = {
    for {
      (parseResult, spec) <- parse()
      resolveResult       <- Future.successful(resolve(parseResult, spec))
      validationReport <- {
        validate(resolveResult, spec)
      }
      lintReport <- lint(resolveResult.baseUnit)
    } yield validationReport.copy(
      results = validationReport.results ++ parseResult.results ++ resolveResult.results ++ lintReport.results
    )
  }

  def validate(document: BaseUnit): Future[AMFValidationReport] = {
    report(document, document.sourceSpec.getOrElse(Spec.AMF))
  }

  def validate(): Future[AMFValidationReport] = {
    for {
      (parseResult, spec) <- parse()
      resolveResult       <- Future.successful(resolve(parseResult, spec))
      validationReport    <- report(parseResult.baseUnit, spec)
    } yield validationReport.copy(
      results = validationReport.results ++ parseResult.results ++ resolveResult.results
    )
  }

  private def report(document: BaseUnit, spec: Spec): Future[AMFValidationReport] = {
    validate(document, spec)
  }

  def lint(): Future[AMFValidationReport] = build().flatMap(b => lint(b.baseUnit))

  def serialize(): Future[String] = build().map(r => serialize(r.baseUnit))

  def serialize(renderOptions: RenderOptions): Future[String] = build().map(r => serialize(r.baseUnit, renderOptions))

  lazy val listDependencies: Seq[Gav] = dependencySet.allDependencies.map(_.descriptor.gav())

}
