/*
 * Copyright © MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.apikit.scaffolding.internal.template;

import org.apache.commons.io.IOUtils;
import org.mule.apikit.scaffolding.api.ScaffoldingConfig;
import org.mule.weave.v2.module.MimeType;
import org.mule.weave.v2.runtime.DataWeaveResult;
import org.mule.weave.v2.runtime.DataWeaveScript;
import org.mule.weave.v2.runtime.DataWeaveScriptingEngine;
import org.mule.weave.v2.runtime.ScriptingBindings;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import static java.lang.String.join;
import static java.nio.charset.StandardCharsets.UTF_8;

public class DataWeaveTemplateEngine implements TemplateEngine {

  private static final String EMPTY_MULE_APP = "<core:mule xmlns:core=\"http://www.mulesoft.org/schema/mule/core\"/>";
  private static final String API_ID = "api";
  private static final String EXISTING_CONFIGURATION_ID = "existingConfiguration";
  private static final String CONFIGURATION_URLS_CSV_ID = "configUrls";

  private final DataWeaveScriptingEngine dataWeaveScriptingEngine;

  public DataWeaveTemplateEngine() {
    this.dataWeaveScriptingEngine = new DataWeaveScriptingEngine();
  }

  @Override
  public Map<String, InputStream> execute(String apiGraph, ScaffoldingConfig scaffoldingConfig)
      throws IOException {
    Map<String, InputStream> results = new HashMap<>();
    ScriptingBindings bindings = getBindings(apiGraph, scaffoldingConfig.getExistingConfigurations());
    for (Map.Entry<String, InputStream> template : scaffoldingConfig.getTemplates().entrySet()) {
      DataWeaveScript compiledScript =
          dataWeaveScriptingEngine.compile(IOUtils.toString(template.getValue(), UTF_8), bindings.bindingNames());
      DataWeaveResult result = compiledScript.write(bindings);
      results.put(getScaffoldedConfigurationFileName(template.getKey()), (InputStream) result.getContent());
    }
    return results;
  }

  private String mergeExistingConfigurations(Set<String> existingConfigURLs) {
    if (existingConfigURLs == null || existingConfigURLs.isEmpty()) {
      return EMPTY_MULE_APP;
    }
    ScriptingBindings bindings = new ScriptingBindings();
    bindings.addBinding(CONFIGURATION_URLS_CSV_ID, join("\n", existingConfigURLs), "text/plain");
    URL mergeScriptURL = DataWeaveTemplateEngine.class.getClassLoader().getResource("scripts/mergeConfigs.dwl");
    DataWeaveScript compiledScript =
        dataWeaveScriptingEngine.compile(mergeScriptURL, bindings.bindingNames());
    DataWeaveResult result = compiledScript.write(bindings);
    return result.getContentAsString();
  }

  private ScriptingBindings getBindings(String apiGraph, Set<String> existingConfigURLs) {
    ScriptingBindings scriptingBindings = new ScriptingBindings();
    scriptingBindings.addBinding(API_ID, apiGraph, MimeType.APPLICATION_JSON().toString());
    scriptingBindings.addBinding(EXISTING_CONFIGURATION_ID, mergeExistingConfigurations(existingConfigURLs), "application/xml");
    return scriptingBindings;
  }

  // TODO: Review output file name. For the time being, just the same name as the template.
  private static String getScaffoldedConfigurationFileName(String templateFileName) {
    return templateFileName.substring(0, templateFileName.indexOf(".")) + ".xml";
  }
}
