/*
 * Copyright © MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.apikit.scaffolding.internal.template;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.net.URI;
import java.util.ArrayList;
import java.util.List;
import org.apache.commons.io.IOUtils;
import org.mule.apikit.scaffolding.api.ScaffoldingConfig;
import org.mule.weave.v2.module.MimeType;
import org.mule.weave.v2.runtime.DataWeaveResult;
import org.mule.weave.v2.runtime.DataWeaveScript;
import org.mule.weave.v2.runtime.DataWeaveScriptingEngine;
import org.mule.weave.v2.runtime.ScriptingBindings;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import static java.nio.charset.StandardCharsets.UTF_8;

public class DataWeaveTemplateEngine implements TemplateEngine {

  private static final String EMPTY_MULE_APP = "<core:mule xmlns:core=\"http://www.mulesoft.org/schema/mule/core\"/>";
  private static final String API_ID = "api";
  private static final String EXISTING_CONFIGURATION_ID = "existingConfiguration";
  private static final String CONFIGURATIONS_CONTENT_ID = "configsContent";
  private static final String API_PATH = "apiPath";

  private final DataWeaveScriptingEngine dataWeaveScriptingEngine;

  public DataWeaveTemplateEngine() {
    this.dataWeaveScriptingEngine = new DataWeaveScriptingEngine();
  }

  @Override
  public Map<String, InputStream> execute(String apiGraph, ScaffoldingConfig scaffoldingConfig)
      throws IOException {
    Map<String, InputStream> results = computeDifferenceUsingTemplateFiles(apiGraph, scaffoldingConfig);

    Map<String, InputStream> mergedResults = addDifferenceToExistingFiles(scaffoldingConfig, results);

    return mergedResults;
  }

  /**
   * generates a XML content for each template specified, with the elements to be added
   * 
   * @param apiGraph parsed API Spec in JSON-LD format
   * @param scaffoldingConfig
   * @return a map, with key = template-file-name and value = xml elements that are not present in the existing mule-application
   * @throws IOException
   */
  private Map<String, InputStream> computeDifferenceUsingTemplateFiles(String apiGraph, ScaffoldingConfig scaffoldingConfig)
      throws IOException {
    Map<String, InputStream> results = new HashMap<>();

    ScriptingBindings bindings = getBindings(apiGraph, scaffoldingConfig.getExistingConfigurations());

    for (Map.Entry<String, InputStream> template : scaffoldingConfig.getTemplates().entrySet()) {
      DataWeaveScript compiledScript =
          dataWeaveScriptingEngine.compile(IOUtils.toString(template.getValue(), UTF_8), bindings.bindingNames());

      DataWeaveResult result = compiledScript.write(bindings);

      results.put(getScaffoldedConfigurationFileName(template.getKey()), (InputStream) result.getContent());
    }
    return results;
  }

  /**
   * Add all the generated elements to existing files, in other cases the contents remains as it is
   * 
   * @param scaffoldingConfig
   * @param results files to be persisted
   * @return
   */
  private Map<String, InputStream> addDifferenceToExistingFiles(ScaffoldingConfig scaffoldingConfig,
                                                                Map<String, InputStream> results) {
    Map<String, InputStream> mergedResults = new HashMap<>();

    results.forEach((k, v) -> mergedResults.put(k, mergeOutput(k, v, scaffoldingConfig.getExistingConfigurations())));
    return mergedResults;
  }

  @Override
  public Map<String, InputStream> generateProperties(String apiPath) {
    Map<String, InputStream> results = new HashMap<>();

    ScriptingBindings bindings = new ScriptingBindings();

    bindings.addBinding(API_PATH, apiPath, "text/plain");

    DataWeaveResult result =
        executeDWScript(DataWeaveTemplateEngine.class.getClassLoader().getResource("scripts/graphql/dev-properties.dwl"),
                        bindings);

    results.put("dev-properties.properties", (InputStream) result.getContent());

    return results;
  }

  private String unionExistingConfigurations(List<String> existingConfigURLs) {
    if (existingConfigURLs == null || existingConfigURLs.isEmpty()) {
      return EMPTY_MULE_APP;
    }

    ScriptingBindings bindings = new ScriptingBindings();

    bindings.addBinding(CONFIGURATIONS_CONTENT_ID, existingConfigURLs, "application/java");

    URL mergeScriptURL = DataWeaveTemplateEngine.class.getClassLoader().getResource("scripts/mergeConfigs.dwl");

    DataWeaveResult result = executeDWScript(mergeScriptURL, bindings);

    return result.getContentAsString();
  }

  private DataWeaveResult executeDWScript(URL scriptURL, ScriptingBindings bindings) {
    DataWeaveScript compiledScript =
        dataWeaveScriptingEngine.compile(scriptURL, bindings.bindingNames());
    return compiledScript.write(bindings);
  }

  private ScriptingBindings getBindings(String apiGraph, Set<String> existingConfigURLs) {
    ScriptingBindings scriptingBindings = new ScriptingBindings();
    scriptingBindings.addBinding(API_ID, apiGraph, MimeType.APPLICATION_JSON().toString());

    List<String> muleXmls = new ArrayList<>();

    existingConfigURLs.forEach(xml -> {
      try {
        muleXmls.add(IOUtils.toString(new FileInputStream(new File(new URI(xml))), UTF_8));
      } catch (Exception e) {
        throw new RuntimeException("Error trying to read existing file", e);
      }
    });

    scriptingBindings.addBinding(EXISTING_CONFIGURATION_ID, unionExistingConfigurations(muleXmls), "application/xml");

    return scriptingBindings;
  }

  // TODO: Review output file name. For the time being, just the same name as the template.
  private static String getScaffoldedConfigurationFileName(String templateFileName) {
    return templateFileName.substring(0, templateFileName.indexOf(".")) + ".xml";
  }

  private InputStream mergeOutput(String fileName, InputStream content, Set<String> existingConfigURLs) {
    try {
      Map<String, String> fileNames = filesNames(existingConfigURLs);

      if (fileNames.get(fileName) == null) {
        return content;
      }

      List<String> filesContentToMerge = new ArrayList<>();

      filesContentToMerge.add(IOUtils.toString(content, UTF_8));

      filesContentToMerge
          .add(IOUtils.toString(new FileInputStream(new File(new URI(fileNames.get(fileName)))), UTF_8));

      return new ByteArrayInputStream(unionExistingConfigurations(filesContentToMerge).getBytes());
    } catch (Exception e) {
      throw new RuntimeException("Error trying to read existing file", e);
    }
  }

  private Map<String, String> filesNames(Set<String> existingConfigURLs) {
    Map<String, String> result = new HashMap<>();
    existingConfigURLs.forEach(url -> result.put(new File(url).getName(), url));
    return result;
  }
}
