/*
 * Copyright © MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.apikit.scaffolding.internal;

import org.mule.apikit.scaffolding.api.Scaffolder;
import org.mule.apikit.scaffolding.api.ScaffoldingConfig;
import org.mule.apikit.scaffolding.api.ScaffoldingResult;
import org.mule.apikit.scaffolding.internal.mapper.ApiGraphMapper;
import org.mule.apikit.scaffolding.internal.template.TemplateEngine;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.io.InputStream;
import java.util.Map;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.singletonList;


public class APIKitScaffolder implements Scaffolder {

  private final ApiGraphMapper graphMapper;
  private final TemplateEngine templateEngine;

  private static final Logger LOGGER = LoggerFactory.getLogger(APIKitScaffolder.class);

  public APIKitScaffolder(ApiGraphMapper graphMapper, TemplateEngine templateEngine) {
    this.graphMapper = graphMapper;
    this.templateEngine = templateEngine;
  }

  public ScaffoldingResult scaffold(ScaffoldingConfig scaffoldingConfig) {
    try {
      Map<String, InputStream> result = getScaffoldingResult(scaffoldingConfig);
      Map<String, String> properties = templateEngine.generateProperties(scaffoldingConfig);
      return new ScaffoldingResultImpl(true, result, emptyList(), properties);
    } catch (IOException e) {
      LOGGER.error(e.getMessage(), e);
      return new ScaffoldingResultImpl(false, emptyMap(),
                                       singletonList(new ScaffoldingErrorImpl(e.getCause(),
                                                                              "An error occurred while scaffolding")));
    }
  }

  public ScaffoldingResult scaffoldMunitTestSuite(ScaffoldingConfig config) {
    try {
      Map<String, InputStream> result = getScaffoldingResult(config);
      return new ScaffoldingResultImpl(true, result, emptyList());
    } catch (IOException e) {
      LOGGER.error(e.getMessage(), e);
      return new ScaffoldingResultImpl(false, emptyMap(),
                                       singletonList(new ScaffoldingErrorImpl(e.getCause(),
                                                                              "An error occurred while scaffolding")));
    }
  }

  private Map<String, InputStream> getScaffoldingResult(ScaffoldingConfig scaffoldingConfig) throws IOException {
    String apiGraph = graphMapper.buildGraph(scaffoldingConfig.getApi());
    return templateEngine.execute(apiGraph, scaffoldingConfig);
  }
}
