%dw 2.0
import mergeWith from dw::core::Objects
input apiPath text/plain
input propertyFiles application/java
output application/java

fun configuresEnvironment(properties) =
    properties.'api.path' != null

/**
* Parse the input files and build a `filename -> contents` mapping for all files that may specify an environment
**/
var parsedEnvironmentFiles =
    propertyFiles
    map { filename: $.filename, contents: readUrl($.contents,"text/x-java-properties") }
    filter configuresEnvironment($.contents)
    reduce (item, acc={}) -> acc ++ { (item.filename): item.contents }

/**
*
* Transforms the values of a given object (keeping the set of keys as-is)
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `object` | { &#40;K&#41;?: V } | The object to map
* | `mapper` | &#40;value: V, key: K, index: Number&#41; &#45;&#62; U | Expression or selector that provides the `key`, `value`, or `index` used for mapping the specified object property value.
* |===
*
* === Example
*
* This example shows how the `mapValues` function behaves for common usage (extracting only a given part of the property values).
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/json
* var commandLineOptions = {
*   '-v': {
*     description: 'Enable/disable verbose mode',
*     aliases: ['--verbose', '--long-messages'],
*     arguments: 0,
*     optional: true
*   },
*   '-o': {
*     description: 'Name of the output file',
*     aliases: ['--output', '--output-file'],
*     arguments: 1,
*     optional: false
*   }
* }
* ---
* commandLineOptions mapValues $.optional
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "-v": true,
*   "-o": false
* }
* ----
**/
fun mapValues <K,V, U>(@StreamCapable object: {(K)?: V}, mapper: (value: V, key: K, index: Number) -> U): {(K)?: U} =
    object mapObject { ($$): mapper($, $$, $$) }

/**
* Describes an object with a single property.
**/
fun property(name, value) = { (name): value }

/**
* Returns an specification of a property that its always added to configuration files
**/
fun alwaysAddProperty(name, value) = (content) -> property(name, value)

/**
* Returns an specification of a property that its added to the configuration files if the given condition holds for the file
**/
fun propertyOnCondition(conditionFn, name, value) =
    (content) ->
        if (conditionFn(content)) property(name, value)
        else {}

/**
* Returns an specification of a property that its added to the configuration file only for new (or empty) files
**/
fun addPropertyOnNewConfig(name, value) =
    propertyOnCondition(isEmpty, name, value)

/**
* Returns an specification of a property that its added to the configuration file only if it was not present before
**/
fun addPropertyIfNotPresent(name, value) =
    propertyOnCondition((content) -> content[name] == null, name, value)

/**
* If no files are given start with an empty `dev-properties.properties`
**/
var environmentFiles =
    if (isEmpty(parsedEnvironmentFiles))
        { 'src/main/resources/dev-properties.properties': {} }
    else
        parsedEnvironmentFiles

/**
* Property specifications for GraphQL APIs
**/
var environmentFile = [
  alwaysAddProperty("api.path", apiPath),
  addPropertyOnNewConfig("http.listener.path", "/graphql"),
  addPropertyOnNewConfig("http.listener.host", "0.0.0.0"),
  addPropertyOnNewConfig("http.listener.port", "8081"),
]

/**
* Runs the given property specifications on the already parsed content
**/
fun addRequiredProperties(content, propertySpecifications) =
    propertySpecifications reduce (spec, acc=content) -> acc mergeWith spec(content)

---
environmentFiles
mapValues addRequiredProperties($, environmentFile)
mapValues write($, 'text/x-java-properties')
