/*
 * Copyright © MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.apikit.scaffolding.internal;

import org.mule.apikit.scaffolding.api.Scaffolder;
import org.mule.apikit.scaffolding.api.ScaffoldingConfig;
import org.mule.apikit.scaffolding.api.ScaffoldingResult;
import org.mule.apikit.scaffolding.internal.mapper.ApiGraphMapper;
import org.mule.apikit.scaffolding.internal.mapper.ApiGraphResult;
import org.mule.apikit.scaffolding.internal.template.TemplateEngine;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.io.InputStream;
import java.util.Map;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.singletonList;


public class APIKitScaffolder implements Scaffolder {

  private final ApiGraphMapper graphMapper;
  private final TemplateEngine templateEngine;

  private static final Logger LOGGER = LoggerFactory.getLogger(APIKitScaffolder.class);

  public APIKitScaffolder(ApiGraphMapper graphMapper, TemplateEngine templateEngine) {
    this.graphMapper = graphMapper;
    this.templateEngine = templateEngine;
  }

  public ScaffoldingResult scaffold(ScaffoldingConfig scaffoldingConfig) {
    ClassLoader current = Thread.currentThread().getContextClassLoader();
    try {
      // Need to set classloader so DW scripts can reference java code
      Thread.currentThread().setContextClassLoader(APIKitScaffolder.class.getClassLoader());
      ScaffoldingInternalResult result = getScaffoldingResult(scaffoldingConfig);
      return new ScaffoldingResultImpl(true, result.getMuleXmls(), emptyList(), result.getProperties());
    } catch (IOException e) {
      LOGGER.error(e.getMessage(), e);
      return new ScaffoldingResultImpl(false, emptyMap(),
                                       singletonList(new ScaffoldingErrorImpl(e.getCause(),
                                                                              "An error occurred while scaffolding")));
    } finally {
      // Restores original classloader
      Thread.currentThread().setContextClassLoader(current);
    }
  }

  public ScaffoldingResult scaffoldMunitTestSuite(ScaffoldingConfig config) {
    try {
      ScaffoldingInternalResult result = getScaffoldingResult(config);
      return new ScaffoldingResultImpl(true, result.getMuleXmls(), emptyList());
    } catch (IOException e) {
      LOGGER.error(e.getMessage(), e);
      return new ScaffoldingResultImpl(false, emptyMap(),
                                       singletonList(new ScaffoldingErrorImpl(e.getCause(),
                                                                              "An error occurred while scaffolding")));
    }
  }

  private ScaffoldingInternalResult getScaffoldingResult(ScaffoldingConfig scaffoldingConfig) throws IOException {
    ApiGraphResult apiGraphResult = graphMapper.buildGraph(scaffoldingConfig.getApi());
    Map<String, InputStream> muleXmls = templateEngine.execute(apiGraphResult, scaffoldingConfig);
    Map<String, String> properties = templateEngine.generateProperties(apiGraphResult, scaffoldingConfig);
    return new ScaffoldingInternalResult(muleXmls, properties);
  }
}
