%dw 2.0
import * from scripts::modules::ScaffoldingModule
import * from scripts::modules::PropertiesModule
import * from scripts::modules::ApiGraphModule
import * from scripts::asyncapi::AsyncApiModule

var anypointmq = {
    prefix: "anypoint-mq",
    uri: "http://www.mulesoft.org/schema/mule/anypoint-mq"
} as Namespace
ns tls http://www.mulesoft.org/schema/mule/tls

var serverPrefix = "anypointmq.server."
var anypointMQConfig = "Anypoint_MQ_configuration"
var supportedSecuritySchemes: Array<String> = ["oauth2", "userPassword", "X509"]
fun getSchemaLocation(): String = anypointmq.uri ++ " http://www.mulesoft.org/schema/mule/anypoint-mq/current/mule-anypoint-mq.xsd"

/**
* Returns property specifications for Anypoint MQ protocol
*/
fun getEnvironmentFile(api, apiPath) = do {
    var servers = getServersForProtocol(api, "anypointmq") filter hasOperationForServer(api, $."@id")
    ---
    getServerEnvParams(api, servers, serverPrefix)
    ++ getConnectionParams(servers)
}

/**
* The params clientAppId and clientSecret are required for Anypoint MQ configuration.
* We will use placeholders for these, and users will be required to complete them in the properties file.
**/
fun getConnectionParams(servers) =
    flatten(
        servers
        map ((server, indx) -> (
            [
                addPropertyIfNotPresent(serverPrefix ++ server."core:name" ++ ".clientAppId", ""),
                addPropertyIfNotPresent(serverPrefix ++ server."core:name" ++ ".clientSecret", "")
            ]
        ))
    )

//TODO: (Improvement) Only scaffold "anypoint-mq-configs" IF at least 1 sub OR pub operation for anypointmq protocol exists
fun scaffoldAsyncApiConfig(api) = do {
    apikitEda#"anypoint-mq-configs": {
        (
            getServersForProtocol(api, "anypointmq") 
            filter hasOperationForServer(api, $."@id")
            map (
                apikitEda#"anypoint-mq-config" @(
                    "serverKey": $."core:name",
                    "configRef": anypointMQConfig ++ "_" ++ $."core:name"
                    ): {}
            )
        )
    }
}

fun scaffoldConfigs(api, existingConfiguration) = {(
    getServersForProtocol(api, "anypointmq")
    filter hasOperationForServer(api, $."@id")
    map do {
        var schemes = getSecuritySchemesForServer(api, $)
        var securityScheme = chooseSecurityScheme(schemes)
        var configName = anypointMQConfig ++ "_" ++ $."core:name"
        ---
        scaffoldElement(
            (anypointmq#"config" @("name": configName): {
                (generateConnectionTemplate($, securityScheme))
            }),
            lookupElementFromApp(
                existingConfiguration,
                (value, key, index) -> key ~= "config" and value.@name == configName
            )
        )
    }
)}

/*
* Prioritize and pick one of the supported security schemes first.
* If not, choose the first security scheme declared from the list.
*/
fun chooseSecurityScheme(schemes) =  do {
    var supportedSchemes = schemes filter (supportedSecuritySchemes contains $)
    ---
    if (!isEmpty(supportedSchemes)) (
        supportedSchemes[0]."security:type"
    )
    else (
        schemes[0]."security:type"
    )
}

fun generateConnectionTemplate(server, securityScheme) = do {
    var clientIdParam = "\${" ++ serverPrefix ++ server."core:name" ++ ".clientAppId}"
    var clientSecretParam = "\${" ++ serverPrefix ++ server."core:name" ++ ".clientSecret}"
    var serverUrl = server."core:urlTemplate"
    var prefix = serverPrefix ++ server."core:name"
    ---
    anypointmq#"connection" @(
        "clientId": clientIdParam,
        "clientSecret": clientSecretParam,
        "url": prefixVariables(serverUrl, prefix)): {
            (generateSecurityTemplate(securityScheme))
    }
}

fun generateSecurityTemplate(securityScheme) =
    securityScheme match {
        case schemeType if (schemeType ~= "userPassword") ->
            anypointmq#"proxy" @(
                "host": "proxy.host.name",
                "port": "0000",
                "username": "dummy",
                "password": "dummy"): {}
        case schemeType if (schemeType ~= "X509") ->
            tls#"context": {
                tls#"trust-store" @("path": "path/to/trustore"): {},
                tls#"key-store" @("path": "path/to/keystore"): {}
            }
        else -> {}
    }
