%dw 2.0
import * from scripts::modules::ScaffoldingModule
import * from scripts::modules::PropertiesModule
import * from scripts::modules::ApiGraphModule
import * from scripts::asyncapi::AsyncApiModule

var solace = {
    prefix: "solace",
    uri: "http://www.mulesoft.org/schema/mule/solace"
} as Namespace
ns tls http://www.mulesoft.org/schema/mule/tls

var serverPrefix = "solace.server."
var bindingPrefix = "solace.binding."
var solaceConfig = "Solace_configuration"
var supportedSecuritySchemes: Array<String> = ["oauth2", "userPassword", "X509"]
fun getSchemaLocation(): String = solace.uri ++ " http://www.mulesoft.org/schema/mule/solace/current/mule-solace.xsd"

/**
* Returns property specifications for Solace protocol
*/
fun getEnvironmentFile(api, apiPath) = do {
    var servers = getServersForProtocol(api, "solace") filter hasOperationForServer(api, $."@id")
    ---
    getServerEnvParams(api, servers, serverPrefix)
    ++ getConnectionParams(servers, api)
}

/**
* The params clientUserName and password are required for Solace configuration.
* msgVpn will be included if it exists in the server bindings
* We will use placeholders for these, and users will be required to complete them in the properties file.
**/
fun getConnectionParams(servers, api) =
    flatten(
        servers
        map ((server, indx) -> (
            chooseSecurityScheme(getSecuritySchemesForServer(api, server)) match {
                case schemeType if (schemeType ~= "userPassword") -> [
                    addPropertyIfNotPresent(serverPrefix ++ server."core:name" ++ ".clientUserName", ""),
                    addPropertyIfNotPresent(serverPrefix ++ server."core:name" ++ ".password", ""),
                    addPropertyIfNotPresent(bindingPrefix ++ server."core:name" ++ ".msgVpn", getMsgVpnFromBinding(api, server)),
                ]
                else -> [
                    addPropertyIfNotPresent(serverPrefix ++ server."core:name" ++ ".clientUserName", ""),
                    addPropertyIfNotPresent(serverPrefix ++ server."core:name" ++ ".password", ""),
                    addPropertyIfNotPresent(bindingPrefix ++ server."core:name" ++ ".msgVpn", getMsgVpnFromBinding(api, server)),
                    addPropertyIfNotPresent(bindingPrefix ++ server."core:name" ++ ".tokenProviderUrl", ""),
                    addPropertyIfNotPresent(bindingPrefix ++ server."core:name" ++ ".clientId", ""),
                    addPropertyIfNotPresent(bindingPrefix ++ server."core:name" ++ ".clientSecret", "")
                ]
            }
        ))
    )

fun getMsgVpnFromBinding(api, server) = do {
    var bindings = getBindingsForServer(api, server)
    var binding = bindings[0]
    var bindingID = binding."@id"
    var payload = (api."@graph" getObjectById bindingID)[0]
    ---
    payload."apiBinding:msgVpn" default ""
}

fun scaffoldAsyncApiConfig(api) = do {
    var servers =  getServersForProtocol(api, "solace") filter hasOperationForServer(api, $."@id")
    ---
    if (!isEmpty(servers)) {
        apikitEda#"solace-configs": {
            (
                servers map
                (
                    apikitEda#"solace-config" @(
                        "serverKey": $."core:name",
                        "configRef": solaceConfig ++ "_" ++ $."core:name"
                        ): {}
                )
            )
        }
    } else {}
}

fun scaffoldConfigs(api, existingConfiguration) = {(
    getServersForProtocol(api, "solace")
    filter hasOperationForServer(api, $."@id")
    map do {
        var schemes = getSecuritySchemesForServer(api, $)
        var securityScheme = chooseSecurityScheme(schemes)
        var configName = solaceConfig ++ "_" ++ $."core:name"
        ---
        scaffoldElement(
            (solace#"config" @("name": configName): {
                (generateConnectionTemplate($, securityScheme))
            }),
            lookupElementFromApp(
                existingConfiguration,
                (value, key, index) -> key ~= "config" and value.@name == configName
            )
        )
    }
)}

/*
* Prioritize and pick one of the supported security schemes first.
* If not, choose the first security scheme declared from the list.
*/
fun chooseSecurityScheme(schemes) =  do {
    var supportedSchemes = schemes filter (supportedSecuritySchemes contains $)
    ---
    if (!isEmpty(supportedSchemes)) (
        supportedSchemes[0]."security:type"
    )
    else (
        schemes[0]."security:type"
    )
}

fun generateConnectionTemplate(server, securityScheme) = do {
    var clientUserName = "\${" ++ serverPrefix ++ server."core:name" ++ ".clientUserName}"
    var password = "\${" ++ serverPrefix ++ server."core:name" ++ ".password}"
    var brokerHost = server."core:urlTemplate"
    var prefix = serverPrefix ++ server."core:name"
    var msgVPN = "\${" ++ bindingPrefix ++ server."core:name" ++ ".msgVpn}"
    ---
    solace#"connection" @(
        "clientUserName": clientUserName,
        "password": password,
        "brokerHost": prefixVariables(brokerHost, prefix),
        "msgVPN": msgVPN
    ): {
        (generateSecurityTemplate(securityScheme, server))
    }
}

/**
* Appends the following configuration to the Solace template
* X509: Will include TLS and OAuth configuration
* OAuth: Will include OAuth configuration
* userPassword: Will not add elements to the Solace configuration
**/
fun generateSecurityTemplate(securityScheme, server) =
    securityScheme match {
        case schemeType if (schemeType ~= "userPassword") -> {}
        case schemeType if (schemeType ~= "X509") -> {
            tls#"context": {
                tls#"trust-store" @("path": "path/to/trustore"): {},
                tls#"key-store" @("path": "path/to/keystore"): {}
            },
            (generateSolaceOauthConfiguration(server))
        }
        else -> {
            (generateSolaceOauthConfiguration(server))
        }
    }

fun generateSolaceOauthConfiguration(server) = do {
    var tokenProviderUrl = "\${" ++ serverPrefix ++ server."core:name" ++ ".tokenProviderUrl}"
    var clientId = "\${" ++ serverPrefix ++ server."core:name" ++ ".clientId}"
    var clientSecret = "\${" ++ serverPrefix ++ server."core:name" ++ ".clientSecret}"
    ---
    solace#"oauth-configuration": {
        solace#"oauth-client-credentials-parameter" @("tokenProviderUrl": tokenProviderUrl, "clientId": clientId, "clientSecret": clientSecret): {
            solace#"oauth-token-refresh-buffer-time": {}
        },
    }
}
