/*
 * Copyright © MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.apikit.scaffolding.internal;

import org.mule.apikit.scaffolding.api.Scaffolder;
import org.mule.apikit.scaffolding.api.ScaffoldingConfig;
import org.mule.apikit.scaffolding.api.ScaffoldingDependency;
import org.mule.apikit.scaffolding.api.ScaffoldingResult;
import org.mule.apikit.scaffolding.internal.error.TemplateEngineError;
import org.mule.apikit.scaffolding.internal.mapper.ApiGraphMapper;
import org.mule.apikit.scaffolding.internal.mapper.ApiGraphResult;
import org.mule.apikit.scaffolding.internal.property.ApplicationProperties;
import org.mule.apikit.scaffolding.internal.property.domain.ProtocolProperty;
import org.mule.apikit.scaffolding.internal.template.TemplateEngine;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.io.InputStream;
import java.util.List;
import java.util.Map;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.toList;


public class APIKitScaffolder implements Scaffolder {

  private final ApiGraphMapper graphMapper;
  private final TemplateEngine templateEngine;

  private static final Logger LOGGER = LoggerFactory.getLogger(APIKitScaffolder.class);

  public APIKitScaffolder(ApiGraphMapper graphMapper, TemplateEngine templateEngine) {
    this.graphMapper = graphMapper;
    this.templateEngine = templateEngine;
  }

  public ScaffoldingResult scaffold(ScaffoldingConfig scaffoldingConfig) {
    ClassLoader current = Thread.currentThread().getContextClassLoader();
    try {
      // Need to set classloader so DW scripts can reference java code
      Thread.currentThread().setContextClassLoader(APIKitScaffolder.class.getClassLoader());
      ScaffoldingInternalResult result = getScaffoldingResult(scaffoldingConfig);
      return new ScaffoldingResultImpl(true, result.getMuleXmls(), emptyList(), result.getProperties(),
                                       getDependencyGAVs(result.getDependencies()), result.getAdditionalInformation());
    } catch (IOException | TemplateEngineError e) {
      LOGGER.error(e.getMessage(), e);
      return new ScaffoldingResultImpl(false, emptyMap(),
                                       singletonList(new ScaffoldingErrorImpl(e.getCause(),
                                                                              e.getMessage())));
    } finally {
      // Restores original classloader
      Thread.currentThread().setContextClassLoader(current);
    }
  }

  public ScaffoldingResult scaffoldMunitTestSuite(ScaffoldingConfig config) {
    ClassLoader current = Thread.currentThread().getContextClassLoader();
    try {
      // Need to set classloader so DW scripts can reference java code
      Thread.currentThread().setContextClassLoader(APIKitScaffolder.class.getClassLoader());
      ScaffoldingInternalResult result = getScaffoldingResult(config);
      return new ScaffoldingResultImpl(true, result.getMuleXmls(), emptyList());
    } catch (IOException | TemplateEngineError e) {
      LOGGER.error(e.getMessage(), e);
      return new ScaffoldingResultImpl(false, emptyMap(),
                                       singletonList(new ScaffoldingErrorImpl(e.getCause(),
                                                                              "An error occurred while scaffolding Munit Test Suite")));
    } finally {
      // Restores original classloader
      Thread.currentThread().setContextClassLoader(current);
    }
  }

  private ScaffoldingInternalResult getScaffoldingResult(ScaffoldingConfig scaffoldingConfig) throws IOException,
      TemplateEngineError {
    ApiGraphResult apiGraphResult = graphMapper.buildGraph(scaffoldingConfig.getApi());
    Map<String, InputStream> muleXmls = templateEngine.execute(apiGraphResult, scaffoldingConfig);
    Map<String, String> properties = templateEngine.generateProperties(apiGraphResult, scaffoldingConfig);
    List<String> dependencies = templateEngine.getDependencies(apiGraphResult, scaffoldingConfig);
    Map<String, List<String>> additionalInformation = templateEngine.getAdditionalInformation(apiGraphResult, scaffoldingConfig);
    return new ScaffoldingInternalResult(muleXmls, properties, dependencies, additionalInformation);
  }

  private List<ScaffoldingDependency> getDependencyGAVs(List<String> dependencies) throws IOException {
    Map<String, ProtocolProperty> protocols = ApplicationProperties.getProperties().getAsyncapi().getProtocols();
    return dependencies.stream().map(dependency -> {
      ProtocolProperty property = protocols.get(dependency);
      return new ScaffoldingDependencyImpl(property.toGav(), property.getClassifier());
    }).collect(toList());
  }
}
