%dw 2.0
import * from scripts::modules::ScaffoldingModule
import * from scripts::modules::PropertiesModule
import * from scripts::modules::ApiGraphModule
import * from scripts::asyncapi::AsyncApiModule

import firstWith from dw::core::Arrays

ns kafka http://www.mulesoft.org/schema/mule/kafka
ns tls http://www.mulesoft.org/schema/mule/tls

var serverPrefix = "kafka.server."
var bindingPrefix = "kafka.binding."
var kafkaProducerConfig = "Apache_Kafka_Producer_configuration"
var kafkaConsumerConfig = "Apache_Kafka_Consumer_configuration"
var userPassConnAttributes = {"username": "dummy", "password": "dummy"}
var supportedSecuritySchemes: Array<String> = ["userPassword", "plain", "scramSha256", "scramSha512", "gssapi", "X509"]

fun getSchemaLocation(): String = kafka.uri ++ " http://www.mulesoft.org/schema/mule/kafka/current/mule-kafka.xsd"

/**
* Returns property specifications for Kafka protocol
*/
fun getEnvironmentFile(api, apiPath) = do {
    var servers = getServersForProtocol(api, "kafka") filter hasOperationForServer(api, $."@id")
    ---
    getServerEnvParams(api, servers, serverPrefix)
    ++ getBindingEnvsParams(api, servers)
}

/*
 * 1) Find publish operations with Kafka groupId binding
 * 2) Derive channels from these operations
 * 3) Loop over servers and check whether these channels are assigned to that server
 * 4) If so, fetch the groupId default binding value from the channel's publish operation binding
*/
fun getBindingEnvsParams(api, servers) = do {
    var operations = getPubOperationsWithProtocolBinding(api, "kafka", "groupId")
    var channels = operations map getEndpointFromPublishOpId(api, $."@id")
    ---
    if (!isEmpty(operations))
    (
        flatten(
            servers
            filter ((item, index) -> !isEmpty(channels firstWith ((channel, index) -> flattenIds(channel."apiContract:server") contains item."@id")))
            map ((server, indx) -> do {
            var channel = channels firstWith ((channel, index) -> flattenIds(channel."apiContract:server") contains server."@id")
            var operation = getEndpointOperations(api, channel) firstWith $."apiContract:method" == "publish"
            var bindingObj = getFirstOpBindingForProtocol(api, operation, "kafka", "groupId")
            ---
            addPropertyIfNotPresent(
                    bindingPrefix ++ server."core:name" ++ ".groupId", 
                    getDefaultValueFromSchema(api, getBindingSchema(api, bindingObj, "groupId"))
            )
            })
        )
    ) else ([])
}

fun scaffoldAsyncApiConfig(api) = do {
    var servers = getServersForProtocol(api, "kafka") filter hasOperationForServer(api, $."@id")
    ---
    if (!isEmpty(servers)) {
        apikitEda#"kafka-configs": {
            (
                servers map
                do {
                    var hasConsumerConfig = hasPublishOperationForServer(api, $."@id")
                    var hasProducerConfig = hasSubscribeOperationForServer(api, $."@id")
                    ---
                    apikitEda#"kafka-config" @(
                        "serverKey": $."core:name",
                        (if (hasProducerConfig) "producerConfigRef": kafkaProducerConfig ++ "_" ++ $."core:name" else {}),
                        (if (hasConsumerConfig) "consumerConfigRef": kafkaConsumerConfig ++ "_" ++ $."core:name" else {})
                        ): {}
                }
            )
        }
    } else {}
}

fun scaffoldConfigs(api, existingConfiguration) = do {
    var servers = getServersForProtocol(api, "kafka")
    ---
    {(
        scaffoldProducerConfigs(api, servers, existingConfiguration)
    ),
    (
        scaffoldConsumerConfigs(api, servers, existingConfiguration)
    )}
}

fun scaffoldProducerConfigs(api, servers, existingConfiguration) = do {
    var producerServers = servers filter hasSubscribeOperationForServer(api, $."@id")
    ---
    producerServers
        map do {
            var schemes = getSecuritySchemesForServer(api, $)
            var kafkaSecurityScheme = chooseSecurityScheme(schemes)
            var producerConfigName = kafkaProducerConfig ++ "_" ++ $."core:name"
            ---
            scaffoldElement(
                (kafka#"producer-config" @("name": producerConfigName): {
                    (generateProducerConnectionTypeTemplate(kafkaSecurityScheme, $))
                }),
                lookupElementFromApp(
                    existingConfiguration,
                    (value, key, index) -> key ~= "producer-config" and value.@name == producerConfigName
                )
            )
        }
}

/*
 * 1) Fetch channels with publish operations
 * 2) Fetch operation IDs that has Kafka groupId binding
 * 3) Use the #1 filter servers that are assigned to channels with publish operation
 * 4) Use #2 to determine whether each server has a groupId assigned based on their channel assignments
*/
fun scaffoldConsumerConfigs(api, servers, existingConfiguration) = do {
    var channelsWithPublishOp = getPublishOpTypes(api) map getEndpointFromPublishOpId(api, $."@id")
    var opIdsWithGroupId = getPubOperationsWithProtocolBinding(api, "kafka", "groupId") map $."@id"
    ---
    servers 
        map do {
            var consumerConfigName = kafkaConsumerConfig ++ "_" ++ $."core:name"
            var existingElement = lookupElementFromApp(
                existingConfiguration,
                (value, key, index) -> key ~= "consumer-config" and value.@name == consumerConfigName
            )
            var channels = channelsWithPublishOp filter (channel, index) -> flattenIds(channel."apiContract:server") contains $."@id"
            var hasGroupId = !isEmpty(
                    channels firstWith 
                    sizeOf(flattenIds($."apiContract:supportedOperation")) > sizeOf(flattenIds($."apiContract:supportedOperation") -- opIdsWithGroupId)
                )
            ---
            if (!isEmpty(channels)) {
                kafka#"consumer-config" @("name": consumerConfigName): {(
                    if (isEmpty(existingElement)) {
                        (generateConsumerConnectionTypeTemplate(api, $, channels, hasGroupId))
                    } else {
                        (updateChannelsAndBindings(existingElement[0], $, channels, hasGroupId, api))
                    }
                )}
            } else { (null) }
        }
}

/*
 * Updates the existing kafka connection XML
 * 1) Add groupId binding if it exists in spec but connection XML does not have it
 * 2) Updates/adds the existing channels (kafka topics)
*/
fun updateChannelsAndBindings(connElement, server, channels, hasGroupId, api) = do {
    var groupIdParam = "\${" ++ bindingPrefix ++ server."core:name" ++ ".groupId}"
    ---
    connElement mapObject ((value, key, index) ->
        if ((key contains "connection")) {
            (key) @(
                    (key.@), 
                    (if (hasGroupId and !key.@.groupId?) "groupId": groupIdParam else {})
                ): addChannels(value, channels)
        } else {
            (key): value
        }
    )
}

fun addChannels(element, channels) = do {
    element mapObject ((value, key, index) ->
        if (value is Object and key ~= "topic-patterns") {
            (key): {
                (channels map kafka#"topic-pattern" @(value: regexChannelVariables($."apiContract:path")): {})
            }
        } else if (value is Object) {
           (key): addChannels(value, channels)
        } else {
            (key): value
        }
    )
}

fun generateProducerConnectionTypeTemplate(securityScheme, server) =
    securityScheme match {
        case schemeType if (schemeType ~= "userPassword") ->
            kafka#"producer-sasl-plain-connection" @((userPassConnAttributes)): {
                (scaffoldBootstrapServers(server))
            }
        case schemeType if (schemeType ~= "plain") ->
            kafka#"producer-sasl-plain-connection" @((userPassConnAttributes)): {
                (scaffoldBootstrapServers(server))
            }
        case schemeType if (schemeType ~= "scramSha256") ->
            kafka#"producer-sasl-scram-connection" @((userPassConnAttributes), "encryptionType": "SCRAM_SHA_256"): {
                (scaffoldBootstrapServers(server))
            }
        case schemeType if (schemeType ~= "scramSha512") ->
            kafka#"producer-sasl-scram-connection" @((userPassConnAttributes), "encryptionType": "SCRAM_SHA_512"): {
                (scaffoldBootstrapServers(server))
            }
        case schemeType if (schemeType ~= "gssapi") ->
            kafka#"producer-sasl-gssapi-connection" @("principal": "dummy", "serviceName": "dummy"): {
                (scaffoldBootstrapServers(server))
            }
        case schemeType if (schemeType ~= "X509") ->
            kafka#"producer-plaintext-connection": {
                tls#"context": {
                    tls#"trust-store" @("path": "path/to/trustore"): {},
                    tls#"key-store" @("path": "path/to/keystore", "type": "jks"): {}
                },
                (scaffoldBootstrapServers(server))
            }
        else -> kafka#"producer-plaintext-connection": { (scaffoldBootstrapServers(server))}
    }

fun generateConsumerConnectionTypeTemplate(api, server, channels, hasGroupId) = do {
    var schemes = getSecuritySchemesForServer(api, server)
    var securityScheme = chooseSecurityScheme(schemes)
    var groupIdParam = "\${" ++ bindingPrefix ++ server."core:name" ++ ".groupId}"
    ---
    securityScheme match {
        case schemeType if (schemeType ~= "userPassword") ->
            kafka#"consumer-sasl-plain-connection" @(
                (userPassConnAttributes),
                (if (hasGroupId) "groupId":groupIdParam else {})): {
                (scaffoldBootstrapServers(server)),
                (scaffoldTopicPatterns(server, channels))
            }
        case schemeType if (schemeType ~= "plain") ->
            kafka#"consumer-sasl-plain-connection" @(
                (userPassConnAttributes),
                (if (hasGroupId) "groupId":groupIdParam else {})): {
                (scaffoldBootstrapServers(server)),
                (scaffoldTopicPatterns(server, channels))
            }
        case schemeType if (schemeType ~= "scramSha256") ->
            kafka#"consumer-sasl-scram-connection" @(
                (userPassConnAttributes), "encryptionType": "SCRAM_SHA_256",
                (if (hasGroupId) "groupId":groupIdParam else {})): {
                (scaffoldBootstrapServers(server)),
                (scaffoldTopicPatterns(server, channels))
            }
        case schemeType if (schemeType ~= "scramSha512") ->
            kafka#"consumer-sasl-scram-connection" @(
                (userPassConnAttributes), "encryptionType": "SCRAM_SHA_512",
                (if (hasGroupId) "groupId":groupIdParam else {})): {
                (scaffoldBootstrapServers(server)),
                (scaffoldTopicPatterns(server, channels))
            }
        case schemeType if (schemeType ~= "gssapi") ->
            kafka#"consumer-sasl-gssapi-connection" @("principal": "dummy", "serviceName": "dummy"
                (if (hasGroupId) "groupId":groupIdParam else {})): {
                (scaffoldBootstrapServers(server)),
                (scaffoldTopicPatterns(server, channels))
            }
        case schemeType if (schemeType ~= "X509") ->
            kafka#"consumer-plaintext-connection"
                @((if (hasGroupId) "groupId":groupIdParam else {})): {
                tls#"context": {
                    tls#"trust-store" @("path": "path/to/trustore"): {},
                    tls#"key-store" @("path": "path/to/keystore", "type": "jks"): {}
                },
                (scaffoldBootstrapServers(server)),
                (scaffoldTopicPatterns(server, channels))
            }
        else -> kafka#"consumer-plaintext-connection"
            @((if (hasGroupId) "groupId":groupIdParam else {})): {
            (scaffoldBootstrapServers(server)),
            (scaffoldTopicPatterns(server, channels))
        }
    }
}

/*
* Prioritize and pick one of the supported security schemes first.
* If not, choose the first security scheme declared from the list.
*/
fun chooseSecurityScheme(schemes) =  do {
    var supportedSchemes = schemes filter (supportedSecuritySchemes contains $)
    ---
    if (!isEmpty(supportedSchemes)) (
        supportedSchemes[0]."security:type"
    )
    else (
        schemes[0]."security:type"
    )
}

fun scaffoldTopicPatterns(server, channels) = do {
    kafka#"topic-patterns": {
        (channels map kafka#"topic-pattern" @(value: regexChannelVariables($."apiContract:path")): {})
    }
}

fun scaffoldBootstrapServers(server) = do {
    var serverURL = server."core:urlTemplate"
    var prefix = serverPrefix ++ server."core:name"
    ---
    kafka#"bootstrap-servers": {
        kafka#"bootstrap-server" @(value: prefixVariables(serverURL, prefix)): {}
    }
}
