/*
 * Copyright © MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.apikit.scaffolding.internal.template;

import org.apache.commons.io.IOUtils;
import org.jdom2.Attribute;
import org.jdom2.Document;
import org.jdom2.Element;
import org.jdom2.JDOMException;
import org.jdom2.Namespace;
import org.jdom2.input.SAXBuilder;
import org.jdom2.output.Format;
import org.jdom2.output.XMLOutputter;
import org.mule.apikit.scaffolding.api.ScaffoldingConfig;
import org.mule.apikit.scaffolding.internal.error.TemplateEngineError;
import org.mule.apikit.scaffolding.internal.mapper.ApiGraphResult;
import org.mule.weave.v2.core.exception.UserException;
import org.mule.weave.v2.runtime.DataWeaveResult;
import org.mule.weave.v2.runtime.DataWeaveScript;
import org.mule.weave.v2.runtime.DataWeaveScriptingEngine;
import org.mule.weave.v2.runtime.ScriptingBindings;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import static java.nio.charset.StandardCharsets.UTF_8;
import static java.util.Arrays.asList;
import static java.util.stream.Collectors.toList;
import static org.jdom2.output.Format.getPrettyFormat;
import static org.jdom2.output.LineSeparator.NL;

public class DataWeaveTemplateEngine implements TemplateEngine {

  private static final String EMPTY_MULE_APP = "<core:mule xmlns:core=\"http://www.mulesoft.org/schema/mule/core\"/>";
  private static final String API_ID = "api";
  private static final String SPEC_ID = "specification";
  private static final String EXISTING_CONFIGURATION_ID = "existingConfiguration";
  private static final String CONFIGURATIONS_CONTENT_ID = "configsContent";
  private static final String API_PATH = "apiPath";
  private static final String PROPERTY_FILES = "propertyFiles";

  private final DataWeaveScriptingEngine dataWeaveScriptingEngine = new DataWeaveScriptingEngine();
  // NOTE: At some point we would probably want to have a generic mechanism for caching compilations
  private final DataWeaveScript propertiesScript = dataWeaveScriptingEngine.compile(getResource("scripts/properties.dwl"),
                                                                                    new String[] {PROPERTY_FILES, API_PATH,
                                                                                        API_ID});
  private final DataWeaveScript mergeScript = dataWeaveScriptingEngine.compile(getResource("scripts/mergeConfigs.dwl"),
                                                                               new String[] {CONFIGURATIONS_CONTENT_ID});

  private final DataWeaveScript protocolsScript =
      dataWeaveScriptingEngine.compile(getResource("scripts/dependencies.dwl"), new String[] {API_ID, API_PATH});

  private final DataWeaveScript additionalInformationScript =
      dataWeaveScriptingEngine.compile(getResource("scripts/additionalInformation.dwl"), new String[] {API_ID});

  private final DataWeaveScript apiFragments =
      dataWeaveScriptingEngine.compile(getResource("scripts/apiFragments.dwl"), new String[] {API_ID});

  @Override
  public Map<String, InputStream> execute(ApiGraphResult apiGraphResult, ScaffoldingConfig scaffoldingConfig)
      throws TemplateEngineError {
    try {
      Map<String, InputStream> results = computeDifferenceUsingTemplateFiles(apiGraphResult, scaffoldingConfig);
      Map<String, InputStream> mergedResults = addDifferenceToExistingFiles(apiGraphResult, scaffoldingConfig, results);
      Map<String, InputStream> namespaceCorrectedResults = fixupNamespaces(mergedResults, scaffoldingConfig);
      return namespaceCorrectedResults;
    } catch (IOException | UserException e) {
      if (e instanceof UserException)
        throw new TemplateEngineError((UserException) e);
      throw new TemplateEngineError(e);
    }
  }

  private Map<String, InputStream> fixupNamespaces(Map<String, InputStream> mergedResults, ScaffoldingConfig config)
      throws IOException {
    Map<String, InputStream> result = new HashMap<>();
    for (Map.Entry<String, InputStream> entry : mergedResults.entrySet()) {
      String filename = entry.getKey();
      InputStream contents = entry.getValue();
      try {
        Document document = new SAXBuilder().build(contents);
        Element root = document.getRootElement();
        addNamespaces(root, root);
        fixupSchemaLocations(filename, root, config);

        // Inefficient, pls fix
        Format format = getPrettyFormat().setLineSeparator(NL);
        String output = new XMLOutputter(format).outputString(document);
        result.put(filename, new ByteArrayInputStream(output.getBytes(UTF_8)));
      } catch (JDOMException e) {
        throw new RuntimeException(e);
      }
    }
    return result;
  }

  private void addNamespaces(Element from, Element to) {
    Namespace ns = from.getNamespace();
    if (!ns.getPrefix().isEmpty()) {
      to.addNamespaceDeclaration(ns);
    }
    from.getChildren().forEach(elem -> this.addNamespaces(elem, to));
  }

  private void fixupSchemaLocations(String filename, Element root, ScaffoldingConfig config) throws IOException,
      JDOMException {
    Set<String> inputConfigs = config.getExistingConfigurations();
    if (inputConfigs.contains(filename)) {
      try (InputStream originalContents = readProjectFile(config, filename)) {
        Document originalDocument = new SAXBuilder().build(originalContents);
        Element originalRoot = originalDocument.getRootElement();

        Namespace xsiNamespace = Namespace.getNamespace("xsi", "http://www.w3.org/2001/XMLSchema-instance");

        Optional<Attribute> originalSchemaLocationAttrOpt =
            Optional.ofNullable(originalRoot.getAttribute("schemaLocation", xsiNamespace));
        Optional<Attribute> rootSchemaLocationAttrOpt = Optional.ofNullable(root.getAttribute("schemaLocation", xsiNamespace));

        if (originalSchemaLocationAttrOpt.isPresent() && rootSchemaLocationAttrOpt.isPresent()) {
          List<String> originalSchemaLocations = parseSchemaLocations(originalSchemaLocationAttrOpt.get().getValue());
          List<String> currentSchemaLocations = parseSchemaLocations(rootSchemaLocationAttrOpt.get().getValue());

          Set<String> combinedSchemaLocations = new LinkedHashSet<>(currentSchemaLocations);
          combinedSchemaLocations.addAll(originalSchemaLocations);

          String newSchemaLocationValue = String.join(" ", combinedSchemaLocations);
          rootSchemaLocationAttrOpt.get().setValue(newSchemaLocationValue);
        }
      }
    }
  }

  private List<String> parseSchemaLocations(String schemaLocationValue) {
    return Arrays.stream(schemaLocationValue.split("\\s+")).collect(Collectors.toList());
  }

  /**
   * generates a XML content for each template specified, with the elements to be added
   *
   * @param apiGraphResult parsed API Spec in JSON-LD format
   * @param config
   * @return a map, with key = template-file-name and value = xml elements that are not present in the existing mule-application
   * @throws IOException
   */
  private Map<String, InputStream> computeDifferenceUsingTemplateFiles(ApiGraphResult apiGraphResult, ScaffoldingConfig config)
      throws IOException, UserException {
    Map<String, InputStream> results = new HashMap<>();

    ScriptingBindings bindings = getBindings(config, apiGraphResult);

    for (Map.Entry<String, InputStream> template : config.getTemplates().entrySet()) {
      DataWeaveScript compiledScript =
          dataWeaveScriptingEngine.compile(IOUtils.toString(template.getValue(), UTF_8), bindings.bindingNames());

      DataWeaveResult result = compiledScript.write(bindings);
      // TODO: Review output file name. For the time being, just the same name as the template.
      results.put(template.getKey(), (InputStream) result.getContent());

    }
    return results;
  }

  /**
   * Add all the generated elements to existing files, in other cases the contents remains as it is
   *
   * @param apiGraphResult
   * @param config
   * @param results files to be persisted
   * @return
   */
  private Map<String, InputStream> addDifferenceToExistingFiles(ApiGraphResult apiGraphResult, ScaffoldingConfig config,
                                                                Map<String, InputStream> results) {
    Set<String> inputConfigs = config.getExistingConfigurations();
    Map<String, InputStream> mergedResults = new HashMap<>();

    for (Map.Entry<String, InputStream> entry : results.entrySet()) {
      String relativePath = entry.getKey();
      InputStream contents = entry.getValue();

      // Check if we have to merge them
      if (inputConfigs.contains(relativePath)) {
        InputStream originalContents = readProjectFile(config, relativePath);
        String mergedContents = unionConfigurations(asList(contents, originalContents), apiGraphResult);
        contents = new ByteArrayInputStream(mergedContents.getBytes());
      }
      mergedResults.put(relativePath, contents);
    }

    return mergedResults;
  }

  private InputStream readProjectFile(ScaffoldingConfig config, String relativePath) {
    String basePath = config.getBasePath();
    File file = Paths.get(basePath, relativePath).toFile();
    InputStream contents;

    try {
      contents = new FileInputStream(file);
    } catch (FileNotFoundException e) {
      throw new RuntimeException("Configuration file does not exist " + file.getAbsolutePath(), e);
    }
    return contents;
  }

  @Override
  public Map<String, String> generateProperties(ApiGraphResult apiGraphResult, ScaffoldingConfig config) {
    List<Object> existingPropertyFiles = config.getExistingResources()
        .stream()
        .filter(v -> v.endsWith(".properties"))
        .map(relativePath -> new Object() {

          public final String filename = relativePath;
          public final InputStream contents = readProjectFile(config, relativePath);
        })
        .collect(toList());

    ScriptingBindings bindings = new ScriptingBindings()
        .addBinding(PROPERTY_FILES, existingPropertyFiles, "application/java")
        .addBinding(API_PATH, config.getApi(), "text/plain")
        .addBinding(API_ID, apiGraphResult.getGraphApi(), "application/json");
    // FIXME: The scaffolding of properties should be parameterizable
    return (Map<String, String>) propertiesScript.write(bindings).getContent();
  }

  @Override
  public List<String> getDependencies(ApiGraphResult apiGraphResult, ScaffoldingConfig scaffoldingConfig) {
    ScriptingBindings bindings = new ScriptingBindings()
        .addBinding(API_ID, apiGraphResult.getGraphApi(), "application/json")
        .addBinding(API_PATH, scaffoldingConfig.getApi(), "text/plain");
    return (List<String>) protocolsScript.write(bindings).getContent();
  }

  @Override
  public Map<String, List<String>> getAdditionalInformation(ApiGraphResult apiGraphResult, ScaffoldingConfig scaffoldingConfig) {
    ScriptingBindings bindings = new ScriptingBindings()
        .addBinding(API_ID, apiGraphResult.getGraphApi(), "application/json");
    return (Map<String, List<String>>) additionalInformationScript.write(bindings).getContent();
  }

  @Override
  public List<String> getApiFragments(String exchangeJson) {
    ScriptingBindings bindings = new ScriptingBindings()
        .addBinding(API_ID, exchangeJson, "application/json");
    return (List<String>) apiFragments.write(bindings).getContent();
  }

  private String unionConfigurations(List<InputStream> existingConfigs, ApiGraphResult apiGraphResult) {
    if (existingConfigs == null || existingConfigs.isEmpty()) {
      return EMPTY_MULE_APP;
    }

    ScriptingBindings bindings = new ScriptingBindings()
        .addBinding(API_ID, apiGraphResult.getGraphApi(), "application/json")
        .addBinding(CONFIGURATIONS_CONTENT_ID, existingConfigs, "application/java")
        .addBinding(SPEC_ID, apiGraphResult.getSpecification(), "text/plain");
    return mergeScript.write(bindings).getContentAsString();
  }

  private ScriptingBindings getBindings(ScaffoldingConfig config, ApiGraphResult apiGraphResult) {
    ScriptingBindings scriptingBindings = new ScriptingBindings();
    scriptingBindings.addBinding(API_ID, apiGraphResult.getGraphApi(), "application/json");

    List<InputStream> muleXMLs = config.getExistingConfigurations().stream()
        .map((relativePath) -> readProjectFile(config, relativePath))
        .collect(toList());

    scriptingBindings.addBinding(EXISTING_CONFIGURATION_ID, unionConfigurations(muleXMLs, apiGraphResult),
                                 "application/xml");

    return scriptingBindings;
  }

  private static URL getResource(String name) {
    return DataWeaveTemplateEngine.class.getClassLoader().getResource(name);
  }
}
