/*
 * Copyright © MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.apikit.scaffolding.internal;

import org.mule.apikit.scaffolding.api.ScaffoldingError;

import java.io.PrintWriter;
import java.io.StringWriter;

import static java.util.Optional.ofNullable;

public class ScaffoldingErrorImpl implements ScaffoldingError {

  private final String message;
  private final Throwable cause;
  private final String stackTrace;
  private final String header;

  public ScaffoldingErrorImpl(String message) {
    this(null, message, null);
  }

  public ScaffoldingErrorImpl(Throwable cause) {
    this(cause, null, null);
  }

  public ScaffoldingErrorImpl(Throwable cause, String message) {
    this(cause, message, null);
  }

  public ScaffoldingErrorImpl(Throwable cause, String message, String header) {
    this.cause = cause;
    this.message = message;
    this.stackTrace = getStackTraceAsString(cause);
    this.header = header;
  }

  @Override
  public String cause() {
    String errorMessage = message != null ? message
        : ofNullable(cause)
            .map(Throwable::getMessage)
            .orElse("An error occurred while scaffolding");

    return errorMessage.endsWith(".") ? errorMessage : errorMessage + ".";
  }

  @Override
  public String stackTrace() {
    if (stackTrace.isEmpty()) {
      return "";
    }
    return header != null ? header + "\n" + stackTrace : stackTrace;
  }

  private String getStackTraceAsString(Throwable throwable) {
    if (throwable == null) {
      return "";
    }
    StringWriter sw = new StringWriter();
    PrintWriter pw = new PrintWriter(sw);
    throwable.printStackTrace(pw);
    return sw.toString();
  }
}
