%dw 2.0
import fail from dw::Runtime
import firstWith from dw::core::Arrays
import * from scripts::modules::ApiGraphModule
import * from scripts::modules::PropertiesModule
import * from scripts::modules::ScaffoldingModule
import getMuleNamespaces from scripts::modules::SharedComponentsModule

var apikitEda = {
    prefix: "apikit-asyncapi",
    uri: "http://www.mulesoft.org/schema/mule/apikit-asyncapi"
} as Namespace
ns doc http://www.mulesoft.org/schema/mule/documentation


var unknownProtocol: String = "unknown"
var asyncApiConfigName = "asynckit-config"
var supportedProtocols: Array<String> = ["kafka", "kafka-secure"]


fun getAsyncApiSchemaLocation(): String = apikitEda.uri ++ " http://www.mulesoft.org/schema/mule/apikit-asyncapi/current/mule-apikit-asyncapi.xsd"

fun getParametersFromEndpoint(api, endpoint) =
    matchingObjectsById(getParameters(api),flattenIds(endpoint."apiContract:parameter"))
    flatMap ((parameter, index) -> parameter."core:name")

fun getOperations(api) =
    flatten(
        getEndpoints(api)
        map ((endpoint, index) ->
            getEndpointOperations(api, endpoint)
            map ((operation, index) ->
                {
                    opType: operation."apiContract:method",
                    channel: endpoint."apiContract:path",
                    parameters: getParametersFromEndpoint(api, endpoint),
                    bindings: getBindingsForOperation(api, operation)
                }
            )
        )
    )

fun hasSubscribeOperations(api) = !isEmpty(getEndpoints(api) map getEndpointOperations(api, $) flatMap $ firstWith $."apiContract:method" == "subscribe")

fun hasPublishOperations(api) = !isEmpty(getEndpoints(api) map getEndpointOperations(api, $) flatMap $ firstWith $."apiContract:method" == "publish")

fun getSubscribeOperations(api) = getOperations(api) filter ((item, index) ->  item.opType == "subscribe")

fun getPublishOperations(api) = getOperations(api) filter ((item, index) -> item.opType == "publish")

fun getPublishOpTypes(api) = api."@graph" filterTypes ["apiContract:Operation"] filter $."apiContract:method"? and $."apiContract:method" == "publish"

fun getEndpointFromPublishOpId(api, publishOpId) = getEndpoints(api) firstWith (flattenIds($."apiContract:supportedOperation") contains publishOpId)

// gets the server objects from api graph
fun getServers(api): Array = api."@graph" filterTypes ["apiContract:Server"]

// get server objects for a protocol from api graph
fun getServersForProtocol(api, protocol: String) = getServers(api) filter ((item, index) -> item."apiContract:protocol" contains protocol)

/*
 * Get unique protocol names from the api graph
 * Returns array of strings
 */
fun getUniqueSupportedProtocolNames(api): Array<String> = do {
    var servers = getServers(api)
    ---
    if (!isEmpty(servers))
    (
        servers map $."apiContract:protocol" distinctBy $ filter ((item, index) -> supportedProtocols contains item)
    )
    else
        fail("Need to specify at least one server in the AsyncAPI Spec")
}

/*
 * This acts like a validation wrapper around getUniqueSupportedProtocolNames
 * function to validate the supported protocols
 */
fun getServerProtocols(api): Array<String> = do {
    var supportedProtocols = getUniqueSupportedProtocolNames(api)
    ---
    if (!isEmpty(supportedProtocols))
        supportedProtocols map if ($ == "kafka-secure") "kafka" else $
    else fail("None of the protocols in the AsyncAPI spec are currently supported")
}

fun replaceInvalidCharacters(channelName) = channelName
    replace "{" with ("(")
    replace "}" with (")")
    replace "/" with ("\\")
    replace "[" with ("((")
    replace "]" with ("))")
    replace "#" with ("@")

/*
 * Returns an xml tag with a format:
 *
 * <?xml version='1.0' encoding='UTF-8'?>
 * <xsi:schemaLocation xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance">
 * http://www.mulesoft.org/schema/mule/core http://www.mulesoft.org/schema/mule/core/current/mule.xsd
 * http://www.mulesoft.org/schema/mule/kafka http://www.mulesoft.org/schema/mule/kafka/current/mule-kafka.xsd
 * http://www.mulesoft.org/schema/mule/anypointmq http://www.mulesoft.org/schema/mule/anypointmq/current/mule-anypointmq.xsd
 * </xsi:schemaLocation>
 */
fun getMuleNamespacesByProtocolSchemas(schemas) =
    if (isEmpty(schemas)) fail("Protocols in the AsyncAPI Spec are currently not supported")
    else
    { 
        ("xmlns:" ++ doc.prefix): doc.uri,
        (getMuleNamespaces(schemas))
    }

fun replaceElementAndAttribute(value:Any, name: String, newValue: Any) = do {
    value match {
        case obj is Object -> obj mapObject ((value, key, index) ->
            if(key ~= name)
                (key): newValue
            else
                (key) @((replaceElementAndAttribute(key.@, name, newValue))): replaceElementAndAttribute(value, name, newValue)
        )
        else -> value
    }
}

fun getPublishBindingsForProtocol(api, protocolName) = do {
    getPublishOpTypes(api)
    filter $."apiBinding:binding"?
    flatMap getBindingsForOperation(api, $)
    filter ($."apiBinding:type" contains protocolName)
    distinctBy $."@id"
}

fun getSubscribeBindingsForProtocol(api, protocolName) = getSubscribeOperations(api) flatMap $."bindings" filter ($."apiBinding:type" contains protocolName)

fun getOperationBindingsForProtocol(operations, protocolName) = operations flatMap $."bindings" filter ($."apiBinding:type" contains protocolName)

fun getDefaultValueFromSchema(api, schema) = 
    if(schema."shacl:defaultValue"?) (
    getObjectById(api."@graph", schema."shacl:defaultValue"."@id")[0]."data:value"
    )
    else if (schema."shacl:in"?) (
    getValueFromEnumList(api, schema)
    )
    else ("")

fun getValueFromEnumList(api, schema) = do {
    var enumListObj = getObjectById(api."@graph", schema."shacl:in"."@id")[0]
    ---
    getObjectById(api."@graph", enumListObj."rdfs:_1"."@id")[0]."data:value"
}

fun getSchemaFromParameter(api, param) = getObjectById(api."@graph", param."raml-shapes:schema"."@id")[0]

fun getDefaultValueFromParam(api, param) = getDefaultValueFromSchema(api, getSchemaFromParameter(api, param))


/*
 * Given a list of server objects, returns a list of (key): value
 * property objects for each server parameter, where 'key' is the 
 * full parameter name, and 'value' is the default value assigned.
*/
fun getServerEnvParams(api, servers, paramPrefix) = do {
    var parameters = getParameters(api)
    ---
    flatten(
        servers
        filter $."apiContract:variable"?
        map ((server, indx) -> matchingObjectsById(parameters, flattenIds(server."apiContract:variable"))
            reduce (
                (serverParamObj, acc = []) -> 
                acc ++ [addPropertyOnNewConfig(
                    paramPrefix ++ server."core:name" ++ "." ++ serverParamObj."core:name", getDefaultValueFromParam(api, serverParamObj))]
            )
        )
    )
}

/*
 * Returns a list of (key): value
 * property objects for each endpoint/channel parameter, where 'key' is the 
 * full parameter name, and 'value' is the default value assigned.
*/
fun getEndpointEnvParams(api) = do {
    var parameters = getParameters(api)
    var endpointParamObjs = getEndpoints(api) 
        filter $."apiContract:parameter"? 
        flatMap matchingObjectsById(parameters, flattenIds($."apiContract:parameter"))
        groupBy $."core:name"
    ---
    endpointParamObjs pluck ((value, key, index) ->
    value firstWith getDefaultValueFromParam(api, $) != "" default value[0])
    reduce (
        (param, acc = []) ->
            acc ++ 
            [addPropertyOnNewConfig("channel." ++ param."core:name", getDefaultValueFromParam(api, param))]
    )
}

fun scaffoldAsyncApiSubscribeFlows(api, existingConfiguration, serverNames: Array<String>, publishOp) = do {
    var channel = getEndpointFromPublishOpId(api, publishOp."@id")
    var flowName = "PUBLISH:" ++ replaceInvalidCharacters(channel."apiContract:path")
    var channelName = prefixVariables(channel."apiContract:path", "channel")
    ---
    scaffoldElement(
        (flow @("name": flowName): {
            apikitEda#"message-listener" @("config-ref": asyncApiConfigName, "channelName": channelName): {
                apikitEda#"servers": {
                    (
                        serverNames map apikitEda#"server" @(value: $): {}
                    )
                }
            },
            logger @("level": "INFO", "message": "#[payload]"): {}
        }),
        lookupElementFromApp(
            existingConfiguration,
            (value, key, index) -> key ~= "flow" and value.@name == flowName
        )
    )
}
