%dw 2.0
import * from scripts::modules::ScaffoldingModule
import * from scripts::modules::PropertiesModule
import * from scripts::modules::ApiGraphModule
import * from scripts::asyncapi::AsyncApiModule

ns kafka http://www.mulesoft.org/schema/mule/kafka
ns tls http://www.mulesoft.org/schema/mule/tls

var serverPrefix = "kafka.server."
var bindingPrefix = "kafka.binding."
var kafkaProducerConfig = "Apache_Kafka_Producer_configuration"
var kafkaConsumerConfig = "Apache_Kafka_Consumer_configuration"
var supportedSecuritySchemes: Array<String> = ["userPassword", "plain", "scramSha256", "scramSha512", "gssapi", "X509"]
fun getSchemaLocation(): String = kafka.uri ++ " http://www.mulesoft.org/schema/mule/kafka/current/mule-kafka.xsd"

/**
* Returns property specifications for Kafka protocol
*/
fun getEnvironmentFile(api, apiPath) = do {
    var servers = getServersForProtocol(api, "kafka")
    ---
    [alwaysAddProperty("api.path", apiPath)]
    ++ getServerEnvParams(api, servers, serverPrefix) 
    ++ getEndpointEnvParams(api)
    ++ getBindingEnvsParams(api, servers)
}

fun getGroupIdBindingSchema(api, binding) = getObjectById(api."@graph", binding."apiBinding:groupId"."@id")[0]

//TODO: Once AsyncAPI 2.6 comes in, that allows specifying servers under each channel,
//      I need to revisit this and handle server selection appropriately
fun getBindingEnvsParams(api, servers) = do {
    var groupIdBindings = getPublishBindingsForProtocol(api, "kafka") filter $."apiBinding:groupId"?
    ---
    if (!isEmpty(groupIdBindings)) 
    (
        flatten(
            servers 
            map ((server, indx) -> addPropertyOnNewConfig(
                    bindingPrefix ++ server."core:name" ++ ".groupId", 
                    getDefaultValueFromSchema(api, getGroupIdBindingSchema(api, groupIdBindings[0])))
            )
        )
    ) else ([])
}

fun scaffoldAsyncApiConfig(api) = do {
    apikitEda#"kafka-configs": {
        (
            getServersForProtocol(api, "kafka") map
                (
                    apikitEda#"kafka-config" @(
                        "serverKey": $."core:name",
                        (if (hasSubscribeOperations(api)) "producerConfigRef": kafkaProducerConfig ++ "_" ++ $."core:name" else {}),
                        (if (hasPublishOperations(api)) "consumerConfigRef": kafkaConsumerConfig ++ "_" ++ $."core:name" else {})
                        ): {}
                )
        )
    }
}

fun scaffoldConfigs(api, existingConfiguration) =
    {(
        if (hasSubscribeOperations(api)) (
            scaffoldProducerConfigs(api, existingConfiguration)
        ) else {}
    ),
    (
        if (hasPublishOperations(api)) (
           scaffoldConsumerConfigs(api, existingConfiguration)
        ) else {}
    )}

fun scaffoldProducerConfigs(api, existingConfiguration) = do {
    getServersForProtocol(api, "kafka")
        map do {
            var schemes = getSecuritySchemesForServer(api, $)
            var kafkaSecurityScheme = chooseSecurityScheme(schemes)
            var producerConfigName = kafkaProducerConfig ++ "_" ++ $."core:name"
            ---
            scaffoldElement(
                (kafka#"producer-config" @("name": producerConfigName): {
                    (generateProducerConnectionTypeTemplate(kafkaSecurityScheme, $))
                }),
                lookupElementFromApp(
                    existingConfiguration,
                    (value, key, index) -> key ~= "producer-config" and value.@name == producerConfigName
                )
            )
        }
}

fun scaffoldConsumerConfigs(api, existingConfiguration) = do {
    getServersForProtocol(api, "kafka")
        map do {
            var consumerConfigName = kafkaConsumerConfig ++ "_" ++ $."core:name"
            ---
            scaffoldElement(
                (kafka#"consumer-config" @("name": consumerConfigName): {
                    (generateConsumerConnectionTypeTemplate(api, $))
                }),
                lookupElementFromApp(
                    existingConfiguration,
                    (value, key, index) -> key ~= "consumer-config" and value.@name == consumerConfigName
                )
            )
        }
}

fun generateProducerConnectionTypeTemplate(securityScheme, server) =
    securityScheme match {
        case schemeType if (schemeType ~= "userPassword") ->
            kafka#"producer-sasl-plain-connection" @("username": "dummy", "password": "dummy"): {
                (scaffoldBootstrapServers(server))
            }
        case schemeType if (schemeType ~= "plain") ->
            kafka#"producer-sasl-plain-connection" @("username": "dummy", "password": "dummy"): {
                (scaffoldBootstrapServers(server))
            }
        case schemeType if (schemeType ~= "scramSha256") ->
            kafka#"producer-sasl-scram-connection" @("username": "dummy", "password": "dummy", "encryptionType": "SCRAM_SHA_256"): {
                (scaffoldBootstrapServers(server))
            }
        case schemeType if (schemeType ~= "scramSha512") ->
            kafka#"producer-sasl-scram-connection" @("username": "dummy", "password": "dummy", "encryptionType": "SCRAM_SHA_512"): {
                (scaffoldBootstrapServers(server))
            }
        case schemeType if (schemeType ~= "gssapi") ->
            kafka#"producer-sasl-gssapi-connection" @("principal": "dummy", "serviceName": "dummy"): {
                (scaffoldBootstrapServers(server))
            }
        case schemeType if (schemeType ~= "X509") ->
            kafka#"producer-plaintext-connection": {
                tls#"context": {
                    tls#"trust-store" @("path": "path/to/trustore"): {},
                    tls#"key-store" @("path": "path/to/keystore", "type": "jks"): {}
                },
                (scaffoldBootstrapServers(server))
            }
        else -> kafka#"producer-plaintext-connection": { (scaffoldBootstrapServers(server))}
    }

// TODO: What is the performance implications of calling getPublishOperations(api)?
fun generateConsumerConnectionTypeTemplate(api, server) = do {
    var schemes = getSecuritySchemesForServer(api, server)
    var securityScheme = chooseSecurityScheme(schemes)
    var pubOps = getPublishOperations(api)
    var channels = pubOps map $.channel
    var hasGroupIdBinding = !isEmpty(getOperationBindingsForProtocol(pubOps, "kafka") filter $."apiBinding:groupId"?)
    var groupIdParam = "\${" ++ bindingPrefix ++ server."core:name" ++ ".groupId}"
    ---
    securityScheme match {
        case schemeType if (schemeType ~= "userPassword") ->
            kafka#"consumer-sasl-plain-connection" @(
                "username": "dummy", "password": "dummy", 
                (if (hasGroupIdBinding) "groupId":groupIdParam else {})): {
                (scaffoldBootstrapServers(server)),
                (scaffoldTopicPatterns(server, channels))
            }
        case schemeType if (schemeType ~= "plain") ->
            kafka#"consumer-sasl-plain-connection" @(
                 "username": "dummy", "password": "dummy", 
                (if (hasGroupIdBinding) "groupId":groupIdParam else {})): {
                (scaffoldBootstrapServers(server)),
                (scaffoldTopicPatterns(server, channels))
            }
        case schemeType if (schemeType ~= "scramSha256") ->
            kafka#"consumer-sasl-scram-connection" @(
                "username": "dummy", "password": "dummy", "encryptionType": "SCRAM_SHA_256",
                (if (hasGroupIdBinding) "groupId":groupIdParam else {})): {
                (scaffoldBootstrapServers(server)),
                (scaffoldTopicPatterns(server, channels))
            }
        case schemeType if (schemeType ~= "scramSha512") ->
            kafka#"consumer-sasl-scram-connection" @(
                "username": "dummy", "password": "dummy", "encryptionType": "SCRAM_SHA_512",
                (if (hasGroupIdBinding) "groupId":groupIdParam else {})): {
                (scaffoldBootstrapServers(server)),
                (scaffoldTopicPatterns(server, channels))
            }
        case schemeType if (schemeType ~= "gssapi") ->
            kafka#"consumer-sasl-gssapi-connection" @("principal": "dummy", "serviceName": "dummy"
                (if (hasGroupIdBinding) "groupId":groupIdParam else {})): {
                (scaffoldBootstrapServers(server)),
                (scaffoldTopicPatterns(server, channels))
            }
        case schemeType if (schemeType ~= "X509") ->
            kafka#"consumer-plaintext-connection"
                @((if (hasGroupIdBinding) "groupId":groupIdParam else {})): {
                tls#"context": {
                    tls#"trust-store" @("path": "path/to/trustore"): {},
                    tls#"key-store" @("path": "path/to/keystore", "type": "jks"): {}
                },
                (scaffoldBootstrapServers(server)),
                (scaffoldTopicPatterns(server, channels))
            }
        else -> kafka#"consumer-plaintext-connection"
            @((if (hasGroupIdBinding) "groupId":groupIdParam else {})): {
            (scaffoldBootstrapServers(server)),
            (scaffoldTopicPatterns(server, channels))
        }
    }
}

/*
* Prioritize and pick one of the supported security schemes first.
* If not, choose the first security scheme declared from the list.
*/
fun chooseSecurityScheme(schemes) =  do {
    var supportedSchemes = schemes filter (supportedSecuritySchemes contains $)
    ---
    if (!isEmpty(supportedSchemes)) (
        supportedSchemes[0]."security:type"
    )
    else (
        schemes[0]."security:type"
    )
}

fun scaffoldTopicPatterns(server, channels) = do {
    kafka#"topic-patterns": {
        (channels map kafka#"topic-pattern" @(value: prefixVariables($, "channel")): {})
    }
}

fun scaffoldBootstrapServers(server) = do {
    var serverURL = server."core:urlTemplate"
    var prefix = serverPrefix ++ server."core:name"
    ---
    kafka#"bootstrap-servers": {
        kafka#"bootstrap-server" @(value: prefixVariables(serverURL, prefix)): {}
    }
}